/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { AccessToken, HttpStatusCode, encodeCrock, randomBytes } from "@gnu-taler/taler-util";
import {
  Button,
  LocalNotificationBanner,
  useChallengerApiContext,
  useLocalNotificationHandler,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useSessionState } from "../hooks/session.js";

type Props = {
  clientId: string;
  onCreated: (nonce:string) => void;
};
export function Setup({ clientId, onCreated }: Props): VNode {
  const { i18n } = useTranslationContext();
  const [notification, withErrorHandler] = useLocalNotificationHandler();
  const { lib } = useChallengerApiContext();
  const { start } = useSessionState();

  const onStart = withErrorHandler(
    async () => {
      return lib.challenger.setup(clientId, "secret-token:chal-secret" as AccessToken);
    },
    (ok) => {
      start({
        clientId,
        redirectURL: "http://exchange.taler.test:1180/kyc-proof/kyc-provider-wallet",
        state: encodeCrock(randomBytes(32)),
      });

      onCreated(ok.body.nonce);
    },
    (fail) => {
      switch (fail.case) {
        case HttpStatusCode.NotFound:
          return i18n.str`Client doesn't exist.`;
      }
    },
  );

  return (
    <Fragment>
      <LocalNotificationBanner notification={notification} />

      <div class="isolate bg-white px-6 py-12">
        <div class="mx-auto max-w-2xl text-center">
          <h2 class="text-3xl font-bold tracking-tight text-gray-900 sm:text-4xl">
            <i18n.Translate>
              Setup new challenge with client ID: &quot;{clientId}&quot;
            </i18n.Translate>
          </h2>
        </div>
          <div class="mt-10">
            <Button
              type="submit"
              class="block w-full disabled:bg-gray-300 rounded-md bg-indigo-600 px-3.5 py-2.5 text-center text-sm font-semibold text-white shadow-sm hover:bg-indigo-500 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600"
              handler={onStart}
            >
              <i18n.Translate>Start</i18n.Translate>
            </Button>
          </div>
      </div>
    </Fragment>
  );
}
