/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { Amounts } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { format } from "date-fns";
import { ComponentChildren, Fragment, h, VNode } from "preact";
import { StateUpdater, useState } from "preact/hooks";
import emptyImage from "../../../../assets/empty.png";
import {
  FormErrors,
  FormProvider,
} from "../../../../components/form/FormProvider.js";
import { InputCurrency } from "../../../../components/form/InputCurrency.js";
import { InputNumber } from "../../../../components/form/InputNumber.js";
import { AuditorBackend, WithId } from "../../../../declaration.js";
import { dateFormatForSettings, useSettings } from "../../../../hooks/useSettings.js";

type Entity = AuditorBackend.DepositConfirmation.DepositConfirmationDetail & WithId;

interface Props {
  instances: Entity[];
  onDelete: (id: Entity) => void;
  onSelect: (depositConfirmation: Entity) => void;
  onUpdate: (
    id: string,
    data: AuditorBackend.DepositConfirmation.DepositConfirmationDetail,
  ) => Promise<void>;
  onCreate: () => void;
  selected?: boolean;
}

export function CardTable({
  instances,
  onCreate,
  onSelect,
  onUpdate,
  onDelete,
}: Props): VNode {
  const [rowSelection, rowSelectionHandler] = useState<string | undefined>(
    undefined,
  );
  const { i18n } = useTranslationContext();
  return (
    <div class="card has-table">
      <header class="card-header">
        <p class="card-header-title">
          <span class="icon">
            <i class="mdi mdi-shopping" />
          </span>
          <i18n.Translate>Deposit Confirmations</i18n.Translate>
        </p>
        <div class="card-header-icon" aria-label="more options">
          <span
            class="has-tooltip-left"
            data-tooltip={i18n.str`add deposit-confirmation`}
          >
            <button class="button is-info" type="button" onClick={onCreate}>
              <span class="icon is-small">
                <i class="mdi mdi-plus mdi-36px" />
              </span>
            </button>
          </span>
        </div>
      </header>
      <div class="card-content">
        <div class="b-table has-pagination">
          <div class="table-wrapper has-mobile-cards">
            {instances.length > 0 ? (
              <Table
                instances={instances}
                onSelect={onSelect}
                onDelete={onDelete}
                onUpdate={onUpdate}
                rowSelection={rowSelection}
                rowSelectionHandler={rowSelectionHandler}
              />
            ) : (
              <EmptyTable />
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
interface TableProps {
  rowSelection: string | undefined;
  instances: Entity[];
  onSelect: (id: Entity) => void;
  onUpdate: (
    id: string,
    data: AuditorBackend.DepositConfirmation.DepositConfirmationDetail,
  ) => Promise<void>;
  onDelete: (serial_id: Entity) => void;
  rowSelectionHandler: StateUpdater<string | undefined>;
}

function Table({
  rowSelection,
  rowSelectionHandler,
  instances,
  onSelect,
  onUpdate,
  onDelete,
}: TableProps): VNode {
  const { i18n } = useTranslationContext();
  const [settings] = useSettings();
  return (
    <div class="table-container">
      <table class="table is-fullwidth is-striped is-hoverable is-fullwidth">
        <thead>
          <tr>
            <th>
              <i18n.Translate>Image</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Description</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Price per unit</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Taxes</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Sales</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Stock</i18n.Translate>
            </th>
            <th>
              <i18n.Translate>Sold</i18n.Translate>
            </th>
            <th />
          </tr>
        </thead>
        <tbody>
          {instances.map((i) => {

            return (
              <Fragment key={i.id}>
                <tr key="info">
                  <td
                    onClick={() =>
                      rowSelection !== i.id && rowSelectionHandler(i.id)
                    }
                    style={{ cursor: "pointer" }}
                  >
                  </td>

                  <td class="is-actions-cell right-sticky">
                    <div class="buttons is-right">
                      <span
                        class="has-tooltip-bottom"
                        data-tooltip={i18n.str`go to product update page`}
                      >
                        <button
                          class="button is-small is-success "
                          type="button"
                          onClick={(): void => onSelect(i)}
                        >
                          <i18n.Translate>Update</i18n.Translate>
                        </button>
                      </span>
                      <span
                        class="has-tooltip-left"
                        data-tooltip={i18n.str`remove this product from the database`}
                      >
                        <button
                          class="button is-small is-danger"
                          type="button"
                          onClick={(): void => onDelete(i)}
                        >
                          <i18n.Translate>Delete</i18n.Translate>
                        </button>
                      </span>
                    </div>
                  </td>
                </tr>
                {rowSelection === i.id && (
                  <tr key="form">
                    <td colSpan={10}>
                    </td>
                  </tr>
                )}
              </Fragment>
            );
          })}
        </tbody>
      </table>
    </div>
  );
}

interface FastProductUpdate {
  incoming: number;
  lost: number;
  price: string;
}
interface UpdatePrice {
  price: string;
}



function EmptyTable(): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="content has-text-grey has-text-centered">
      <p>
        <span class="icon is-large">
          <i class="mdi mdi-emoticon-sad mdi-48px" />
        </span>
      </p>
      <p>
        <i18n.Translate>
          There is no products yet, add more pressing the + sign
        </i18n.Translate>
      </p>
    </div>
  );
}

function difference(price: string, tax: number) {
  if (!tax) return price;
  const ps = price.split(":");
  const p = parseInt(ps[1], 10);
  ps[1] = `${p - tax}`;
  return ps.join(":");
}