/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { ErrorMessage } from "../../components/ErrorMessage.js";
import { Input, LightText, SubTitle, Title, WarningBox } from "../../components/styled/index.js";
import { TermsOfService } from "../../components/TermsOfService/index.js";
import { Button } from "../../mui/Button.js";
import { State } from "./index.js";
import { assertUnreachable } from "@gnu-taler/taler-util";


export function VerifyView({
  expectedCurrency,
  onCancel,
  onAccept,
  result,
  loading,
  knownExchanges,
  url,
}: State.Verify): VNode {
  const { i18n } = useTranslationContext();

  return (
    <Fragment>
      <section>
        {!expectedCurrency ? (
          <Title>
            <i18n.Translate>Add new exchange</i18n.Translate>
          </Title>
        ) : (
          <SubTitle>
            <i18n.Translate>Add exchange for {expectedCurrency}</i18n.Translate>
          </SubTitle>
        )}
        {!result && (
          <LightText>
            <i18n.Translate>
              Enter the URL of an exchange you trust.
            </i18n.Translate>
          </LightText>
        )}
        {(() => {
          if (!result) return;
          if (result.type == "ok") {
            return <LightText>
              <i18n.Translate>
                An exchange has been found! Review the information and click next
              </i18n.Translate>
            </LightText>
          }
          switch (result.case) {
            case "already-active": {
              return <WarningBox>
                <i18n.Translate>This exchange is already in your list.</i18n.Translate>
              </WarningBox>
            }
            case "invalid-protocol": {
              return <WarningBox>
                <i18n.Translate>Only exchange accessible through "http" and "https" are allowed.</i18n.Translate>
              </WarningBox>
            }
            case "invalid-version": {
              return <WarningBox>
                <i18n.Translate>This exchange protocol version is not supported: "{result.body}".</i18n.Translate>
              </WarningBox>
            }
            case "invalid-currency": {
              return <WarningBox>
                <i18n.Translate>This exchange currency "{result.body}" doesn&apos;t match the expected currency {expectedCurrency}.</i18n.Translate>
              </WarningBox>
            }
            default: {
              assertUnreachable(result.case)
            }
          }
        })()}
        <p>
          <Input invalid={result && result.type !== "ok"} >
            <label>URL</label>
            <input
              type="text"
              placeholder="https://"
              value={url.value}
              onInput={(e) => {
                if (url.onInput) {
                  url.onInput(e.currentTarget.value)
                }
              }}
            />
          </Input>
          {loading && (
            <div>
              <i18n.Translate>loading</i18n.Translate>...
            </div>
          )}
          {result && result.type === "ok" && (
            <Fragment>
              <Input>
                <label>
                  <i18n.Translate>Version</i18n.Translate>
                </label>
                <input type="text" disabled value={result.body.version} />
              </Input>
              <Input>
                <label>
                  <i18n.Translate>Currency</i18n.Translate>
                </label>
                <input type="text" disabled value={result.body.currency} />
              </Input>
            </Fragment>
          )}
        </p>
        {url.value && url.error && (
          <ErrorMessage
            title={i18n.str`Can't use the URL: "${url.value}"`}
            description={url.error}
          />
        )}
      </section>
      <footer>
        <Button variant="contained" color="secondary" onClick={onCancel}>
          <i18n.Translate>Cancel</i18n.Translate>
        </Button>
        <Button
          variant="contained"
          disabled={
            !result ||
            result.type !== "ok"
          }
          onClick={onAccept}
        >
          <i18n.Translate>Next</i18n.Translate>
        </Button>
      </footer>
      <section>
        <ul>
          {knownExchanges.map(ex => {
            return <li><a href="#" onClick={(e) => {
              if (url.onInput) {
                url.onInput(ex.href)
              }
              e.preventDefault()
            }}>
              {ex.href}</a></li>
          })}
        </ul>
      </section>
    </Fragment>
  );
}


export function ConfirmView({
  url,
  onCancel,
  onConfirm,
}: State.Confirm): VNode {
  const { i18n } = useTranslationContext();

  return (
    <Fragment>
      <section>
        <Title>
          <i18n.Translate>Review terms of service</i18n.Translate>
        </Title>
        <div>
          <i18n.Translate>Exchange URL</i18n.Translate>:
          <a href={url} target="_blank" rel="noreferrer">
            {url}
          </a>
        </div>
      </section>


      <TermsOfService key="terms" exchangeUrl={url} >
        <footer>
          <Button
            key="cancel"
            variant="contained"
            color="secondary"
            onClick={onCancel}
          >
            <i18n.Translate>Cancel</i18n.Translate>
          </Button>
          <Button
            key="add"
            variant="contained"
            color="success"
            onClick={onConfirm}
          >
            <i18n.Translate>Add exchange</i18n.Translate>
          </Button>
        </footer>
      </TermsOfService>
    </Fragment>
  );
}
