/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { NotificationType, WalletNotification } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig, defaultCoinConfig } from "../harness/denomStructures.js";
import {
  ExchangeService,
  FakebankService,
  GlobalTestState,
  WalletClient,
  WalletService,
  setupDb,
} from "../harness/harness.js";
import { withdrawViaBankV2 } from "../harness/helpers.js";

export async function runWalletObservabilityTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await FakebankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const exchangeBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );
  exchange.addBankAccount("1", exchangeBankAccount);

  bank.setSuggestedExchange(exchange, exchangeBankAccount.accountPaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const coinConfig: CoinConfig[] = defaultCoinConfig.map((x) => x("TESTKUDOS"));
  exchange.addCoinConfigList(coinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  const walletService = new WalletService(t, {
    name: "wallet",
    useInMemoryDb: true,
  });
  await walletService.start();
  await walletService.pingUntilAvailable();

  const allNotifications: WalletNotification[] = [];

  const walletClient = new WalletClient({
    name: "wallet",
    unixPath: walletService.socketPath,
    onNotification(n) {
      console.log("got notification", n);
      allNotifications.push(n);
    },
  });
  await walletClient.connect();
  await walletClient.client.call(WalletApiOperation.InitWallet, {
    config: {
      testing: {
        skipDefaults: true,
        emitObservabilityEvents: true,
      },
    },
  });

  const wres = await withdrawViaBankV2(t, {
    amount: "TESTKUDOS:10",
    bank,
    exchange,
    walletClient,
  });
  await wres.withdrawalFinishedCond;

  const requestObsEvents = allNotifications.filter(
    (x) => x.type === NotificationType.RequestObservabilityEvent,
  );
  const taskObsEvents = allNotifications.filter(
    (x) => x.type === NotificationType.TaskObservabilityEvent,
  );

  console.log(`req events: ${requestObsEvents.length}`);
  console.log(`task events: ${taskObsEvents.length}`);

  t.assertTrue(requestObsEvents.length > 30);
  t.assertTrue(taskObsEvents.length > 30);
}

runWalletObservabilityTest.suites = ["wallet"];
