/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
  makeTestPaymentV2,
} from "../harness/helpers.js";
import {
  AbsoluteTime,
  AmountString,
  Duration,
  NotificationType,
  TransactionMajorState,
  TransactionMinorState,
  j2s,
} from "@gnu-taler/taler-util";

/**
 * Test that creates various transactions and exports the resulting
 * database.  Used to generate a database export file for DB compatibility
 * testing.
 */
export async function runWalletGenDbTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV2(t);

  // Withdraw digital cash into the wallet.

  await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:50",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:10",
    fulfillment_url: "taler://fulfillment-success/thx",
  };

  await makeTestPaymentV2(t, { walletClient, merchant, order });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const purseExpiration = AbsoluteTime.toProtocolTimestamp(
    AbsoluteTime.addDuration(
      AbsoluteTime.now(),
      Duration.fromSpec({ days: 2 }),
    ),
  );

  const peerPullIniResp = await walletClient.call(
    WalletApiOperation.InitiatePeerPullCredit,
    {
      exchangeBaseUrl: exchange.baseUrl,
      partialContractTerms: {
        summary: "Hello World",
        amount: "TESTKUDOS:5" as AmountString,
        purse_expiration: purseExpiration,
      },
    },
  );

  const peerPullCreditReadyCond = walletClient.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.TransactionStateTransition &&
      x.transactionId === peerPullIniResp.transactionId &&
      x.newTxState.major === TransactionMajorState.Pending &&
      x.newTxState.minor === TransactionMinorState.Ready,
  );

  await peerPullCreditReadyCond;

  const checkResp = await walletClient.call(
    WalletApiOperation.PreparePeerPullDebit,
    {
      talerUri: peerPullIniResp.talerUri,
    },
  );

  await walletClient.call(WalletApiOperation.ConfirmPeerPullDebit, {
    transactionId: checkResp.transactionId,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
}

runWalletGenDbTest.suites = ["wallet"];
