/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AmountString,
  encodeCrock,
  getRandomBytes,
  j2s,
  TalerError,
} from "@gnu-taler/taler-util";
import {
  applyRunConfigDefaults,
  CryptoDispatcher,
  SynchronousCryptoWorkerFactoryPlain,
} from "@gnu-taler/taler-wallet-core";
import {
  checkReserve,
  depositCoin,
  downloadExchangeInfo,
  findDenomOrThrow,
  refreshCoin,
  topupReserveWithDemobank,
  withdrawCoin,
} from "@gnu-taler/taler-wallet-core/dbless";
import { GlobalTestState, harnessHttpLib } from "../harness/harness.js";
import { createSimpleTestkudosEnvironmentV2 } from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runWalletDblessTest(t: GlobalTestState) {
  // Set up test environment

  const { bank, exchange } = await createSimpleTestkudosEnvironmentV2(t);

  const http = harnessHttpLib;
  const cryptiDisp = new CryptoDispatcher(
    new SynchronousCryptoWorkerFactoryPlain(),
  );
  const cryptoApi = cryptiDisp.cryptoApi;

  try {
    // Withdraw digital cash into the wallet.

    const exchangeInfo = await downloadExchangeInfo(exchange.baseUrl, http);

    const reserveKeyPair = await cryptoApi.createEddsaKeypair({});

    let reserveUrl = new URL(
      `reserves/${reserveKeyPair.pub}`,
      exchange.baseUrl,
    );
    reserveUrl.searchParams.set("timeout_ms", "30000");
    const longpollReq = http.fetch(reserveUrl.href, {
      method: "GET",
    });

    await topupReserveWithDemobank({
      amount: "TESTKUDOS:10" as AmountString,
      http,
      reservePub: reserveKeyPair.pub,
      corebankApiBaseUrl: bank.corebankApiBaseUrl,
      exchangeInfo,
    });

    console.log("waiting for longpoll request");
    const resp = await longpollReq;
    console.log(`got response, status ${resp.status}`);

    console.log(exchangeInfo);

    await checkReserve(http, exchange.baseUrl, reserveKeyPair.pub);

    const defaultConfig = applyRunConfigDefaults();

    const d1 = findDenomOrThrow(exchangeInfo, "TESTKUDOS:8" as AmountString, {
      denomselAllowLate: defaultConfig.testing.denomselAllowLate,
    });

    const coin = await withdrawCoin({
      http,
      cryptoApi,
      reserveKeyPair: {
        reservePriv: reserveKeyPair.priv,
        reservePub: reserveKeyPair.pub,
      },
      denom: d1,
      exchangeBaseUrl: exchange.baseUrl,
    });

    const wireSalt = encodeCrock(getRandomBytes(16));
    const merchantPub = encodeCrock(getRandomBytes(32));
    const contractTermsHash = encodeCrock(getRandomBytes(64));

    await depositCoin({
      contractTermsHash,
      merchantPub,
      wireSalt,
      amount: "TESTKUDOS:4" as AmountString,
      coin: coin,
      cryptoApi,
      exchangeBaseUrl: exchange.baseUrl,
      http,
    });

    // Idempotency
    await depositCoin({
      contractTermsHash,
      merchantPub,
      wireSalt,
      amount: "TESTKUDOS:4" as AmountString,
      coin: coin,
      cryptoApi,
      exchangeBaseUrl: exchange.baseUrl,
      http,
    });

    const refreshDenoms = [
      findDenomOrThrow(exchangeInfo, "TESTKUDOS:1" as AmountString, {
        denomselAllowLate: defaultConfig.testing.denomselAllowLate,
      }),
      findDenomOrThrow(exchangeInfo, "TESTKUDOS:1" as AmountString, {
        denomselAllowLate: defaultConfig.testing.denomselAllowLate,
      }),
    ];

    await refreshCoin({
      oldCoin: coin,
      cryptoApi,
      http,
      newDenoms: refreshDenoms,
    });
  } catch (e) {
    if (e instanceof TalerError) {
      console.log(e);
      console.log(j2s(e.errorDetail));
    } else {
      console.log(e);
    }
    throw e;
  }
}

runWalletDblessTest.suites = ["wallet"];
