/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Amounts,
  Duration,
  MerchantApiClient,
  MerchantContractTerms,
  PreparePayResultType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Test wallet:
 * - balance error messages
 * - different types of insufficient balance.
 *
 * Related bugs:
 * https://bugs.taler.net/n/7299
 */
export async function runWalletBalanceTest(t: GlobalTestState) {
  // Set up test environment

  const { merchant, walletClient, exchange, bank } =
    await createSimpleTestkudosEnvironmentV2(t);

  await merchant.addInstanceWithWireAccount({
    id: "myinst",
    name: "My Instance",
    paytoUris: ["payto://void/foo"],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  const merchantClient = new MerchantApiClient(
    merchant.makeInstanceBaseUrl("myinst"),
  );

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  console.log("withdrawal finished");

  const order: Partial<MerchantContractTerms> = {
    summary: "Buy me!",
    amount: "TESTKUDOS:5",
    fulfillment_url: "taler://fulfillment-success/thx",
  };

  console.log("creating order");

  const orderResp = await merchantClient.createOrder({
    order,
  });

  console.log("created order with merchant");

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  console.log("queried order at merchant");

  // Make wallet pay for the order

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.InsufficientBalance,
  );

  t.assertDeepEqual(
    preparePayResult.status,
    PreparePayResultType.InsufficientBalance,
  );

  t.assertTrue(
    Amounts.isNonZero(
      preparePayResult.balanceDetails.balanceReceiverAcceptable,
    ),
  );

  t.assertTrue(
    Amounts.isZero(preparePayResult.balanceDetails.balanceReceiverDepositable),
  );

  console.log("waiting for transactions to finalize");

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
}

runWalletBalanceTest.suites = ["wallet"];
