/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  ConfirmPayResultType,
  Duration,
  MerchantApiClient,
  PreparePayResultType,
  narrowOpSuccessOrThrow,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Test for taler://payment-template/ URIs
 */
export async function runPaymentTemplateTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV2(t);

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  const createTemplateRes = await merchantClient.createTemplate({
    template_id: "template1",
    template_description: "my test template",
    template_contract: {
      minimum_age: 0,
      pay_duration: Duration.toTalerProtocolDuration(
        Duration.fromSpec({
          minutes: 2,
        }),
      ),
      summary: "hello, I'm a summary",
    },
  });
  narrowOpSuccessOrThrow(createTemplateRes);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });
  await wres.withdrawalFinishedCond;

  // Request a template payment

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForTemplate,
    {
      talerPayTemplateUri: `taler+http://pay-template/localhost:${merchant.port}/template1?amount=TESTKUDOS:1`,
      templateParams: {},
    },
  );

  console.log(preparePayResult);

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.PaymentPossible,
  );

  // Pay for it

  const r2 = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: preparePayResult.transactionId,
  });

  t.assertTrue(r2.type === ConfirmPayResultType.Done);

  // Check if payment was successful.

  const orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: preparePayResult.contractTerms.order_id,
    instance: "default",
  });

  t.assertTrue(orderStatus.order_status === "paid");
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
}

runPaymentTemplateTest.suites = ["wallet"];
