/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Sample fault injection test.
 */

/**
 * Imports.
 */
import { ConfirmPayResultType, MerchantApiClient } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  FaultInjectedExchangeService,
  FaultInjectionRequestContext,
  FaultInjectionResponseContext,
} from "../harness/faultInjection.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  MerchantService,
  generateRandomPayto,
  setupDb,
} from "../harness/harness.js";
import {
  createWalletDaemonWithClient,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runPaymentFaultTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const exchangeBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );

  const faultyExchange = new FaultInjectedExchangeService(t, exchange, 8091);
  // Base URL must contain port that the proxy is listening on.
  await exchange.modifyConfig(async (config) => {
    config.setString("exchange", "base_url", "http://localhost:8091/");
  });

  bank.setSuggestedExchange(
    faultyExchange,
    exchangeBankAccount.accountPaytoUri,
  );

  await bank.start();

  await bank.pingUntilAvailable();

  await exchange.addBankAccount("1", exchangeBankAccount);
  exchange.addOfferedCoins(defaultCoinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  // Print all requests to the exchange
  faultyExchange.faultProxy.addFault({
    async modifyRequest(ctx: FaultInjectionRequestContext) {
      console.log("got request", ctx);
    },
    async modifyResponse(ctx: FaultInjectionResponseContext) {
      console.log("got response", ctx);
    },
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    currency: "TESTKUDOS",
    httpPort: 8083,
    database: db.connStr,
  });

  merchant.addExchange(faultyExchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  await merchant.addInstanceWithWireAccount({
    id: "default",
    name: "Default Instance",
    paytoUris: [generateRandomPayto("merchant-default")],
  });

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  console.log("setup done!");

  const { walletClient } = await createWalletDaemonWithClient(t, {
    name: "default",
  });

  await walletClient.call(WalletApiOperation.GetBalances, {});

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange: faultyExchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  // Set up order.

  const orderResp = await merchantClient.createOrder({
    order: {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "taler://fulfillment-success/thx",
    },
  });

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  // Make wallet pay for the order

  const prepResp = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  // Drop 3 responses from the exchange.
  let faultCount = 0;
  faultyExchange.faultProxy.addFault({
    async modifyResponse(ctx: FaultInjectionResponseContext) {
      console.log(`in modifyResponse for ${ctx.request.requestUrl}`);
      if (
        !ctx.request.requestUrl.endsWith("/deposit") &&
        !ctx.request.requestUrl.endsWith("/batch-deposit")
      ) {
        return;
      }
      if (faultCount < 3) {
        console.log(`blocking /deposit request #${faultCount}`);
        faultCount++;
        ctx.dropResponse = true;
      } else {
        console.log(`letting through /deposit request #${faultCount}`);
      }
    },
  });

  // confirmPay won't work, as the exchange is unreachable

  const confirmPayResp = await walletClient.call(
    WalletApiOperation.ConfirmPay,
    {
      transactionId: prepResp.transactionId,
    },
  );

  t.assertDeepEqual(confirmPayResp.type, ConfirmPayResultType.Pending);

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // Check if payment was successful.

  orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "paid");
}

runPaymentFaultTest.suites = ["wallet"];
runPaymentFaultTest.timeoutMs = 120000;
