/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  ConfirmPayResultType,
  Duration,
  MerchantApiClient,
  MerchantContractTerms,
  PreparePayResultType,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  applyTimeTravelV2,
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Run a test for the following scenario:
 *
 * - Wallet claims an order
 * - Merchant goes down
 * - Wallet tried to pay, but it fails as the merchant is unavailable
 * - The order expires
 * - The merchant goes back up again
 * - Instead of trying to get an abort-refund, the wallet notices that
 *   the order is expired, puts the transaction into "failed",
 *   refreshes allocated coins and thus raises the balance again.
 */
export async function runPaymentExpiredTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV2(t);

  // Withdraw digital cash into the wallet.

  await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // Order that can only be paid within five minutes.
  const order: Partial<MerchantContractTerms> = {
    summary: "Buy me!",
    amount: "TESTKUDOS:5",
    fulfillment_url: "taler://fulfillment-success/thx",
    pay_deadline: AbsoluteTime.toProtocolTimestamp(
      AbsoluteTime.addDuration(
        AbsoluteTime.now(),
        Duration.fromSpec({ minutes: 5 }),
      ),
    ),
  };

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  const orderResp = await merchantClient.createOrder({
    order,
  });

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  t.assertDeepEqual(
    preparePayResult.status,
    PreparePayResultType.PaymentPossible,
  );

  await applyTimeTravelV2(
    Duration.toMilliseconds(Duration.fromSpec({ hours: 1 })),
    { walletClient, exchange, merchant },
  );

  const confirmPayResult = await walletClient.call(
    WalletApiOperation.ConfirmPay,
    { transactionId: preparePayResult.transactionId },
  );
  console.log("confirm pay result:");
  console.log(j2s(confirmPayResult));
  t.assertDeepEqual(confirmPayResult.type, ConfirmPayResultType.Pending);
  await walletClient.call(WalletApiOperation.AbortTransaction, {
    transactionId: preparePayResult.transactionId,
  });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(bal);

  t.assertAmountEquals(bal.balances[0].available, "TESTKUDOS:18.93");

  const txns = await walletClient.call(WalletApiOperation.GetTransactions, {
    includeRefreshes: true,
  });
  console.log(j2s(txns));
}

runPaymentExpiredTest.suites = ["wallet"];
