/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { Duration, j2s } from "@gnu-taler/taler-util";
import { Wallet, WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  MerchantService,
  generateRandomPayto,
  setupDb,
} from "../harness/harness.js";
import {
  createWalletDaemonWithClient,
  makeTestPaymentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runCurrencyScopeTest(t: GlobalTestState) {
  // Set up test environment
  const dbDefault = await setupDb(t);

  const dbExchangeTwo = await setupDb(t, {
    nameSuffix: "exchange2",
  });

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: dbDefault.connStr,
    httpPort: 8082,
  });

  const exchangeOne = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: dbDefault.connStr,
  });

  const exchangeTwo = ExchangeService.create(t, {
    name: "testexchange-2",
    currency: "TESTKUDOS",
    httpPort: 8281,
    database: dbExchangeTwo.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    currency: "TESTKUDOS",
    httpPort: 8083,
    database: dbDefault.connStr,
  });

  const exchangeOneBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );
  await exchangeOne.addBankAccount("1", exchangeOneBankAccount);

  const exchangeTwoBankAccount = await bank.createExchangeAccount(
    "myexchange2",
    "x",
  );
  await exchangeTwo.addBankAccount("1", exchangeTwoBankAccount);

  bank.setSuggestedExchange(
    exchangeOne,
    exchangeOneBankAccount.accountPaytoUri,
  );

  await bank.start();

  await bank.pingUntilAvailable();

  // Set up the first exchange

  exchangeOne.addOfferedCoins(defaultCoinConfig);
  await exchangeOne.start();
  await exchangeOne.pingUntilAvailable();

  // Set up the second exchange

  exchangeTwo.addOfferedCoins(defaultCoinConfig);
  await exchangeTwo.start();
  await exchangeTwo.pingUntilAvailable();

  // Start and configure merchant

  merchant.addExchange(exchangeOne);
  merchant.addExchange(exchangeTwo);

  await merchant.start();
  await merchant.pingUntilAvailable();

  await merchant.addInstanceWithWireAccount({
    id: "default",
    name: "Default Instance",
    paytoUris: [generateRandomPayto("merchant-default")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [generateRandomPayto("minst1")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  const { walletClient } = await createWalletDaemonWithClient(t, {
    name: "wallet",
  });

  console.log("setup done!");

  // Withdraw digital cash into the wallet.

  const w1 = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange: exchangeOne,
    amount: "TESTKUDOS:6",
  });

  const w2 = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange: exchangeTwo,
    amount: "TESTKUDOS:6",
  });

  await w1.withdrawalFinishedCond;
  await w2.withdrawalFinishedCond;

  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(j2s(bal));

  // Separate balances, exchange-scope.
  t.assertDeepEqual(bal.balances.length, 2);

  await walletClient.call(WalletApiOperation.AddGlobalCurrencyExchange, {
    currency: "TESTKUDOS",
    exchangeBaseUrl: exchangeOne.baseUrl,
    exchangeMasterPub: exchangeOne.masterPub,
  });

  await walletClient.call(WalletApiOperation.AddGlobalCurrencyExchange, {
    currency: "TESTKUDOS",
    exchangeBaseUrl: exchangeTwo.baseUrl,
    exchangeMasterPub: exchangeTwo.masterPub,
  });

  const ex = walletClient.call(
    WalletApiOperation.ListGlobalCurrencyExchanges,
    {},
  );
  console.log("global currency exchanges:");
  console.log(j2s(ex));

  const bal2 = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(j2s(bal2));

  // Global currencies are merged
  t.assertDeepEqual(bal2.balances.length, 1);
}

runCurrencyScopeTest.suites = ["wallet"];
