/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import * as tests from "@gnu-taler/web-util/testing";
import { expect } from "chai";
import { MerchantBackend } from "../declaration.js";
import {
  useInstanceReserves,
  useReservesAPI,
} from "./reserves.js";
import { ApiMockEnvironment } from "./testing.js";
import {
  API_CREATE_RESERVE,
  API_DELETE_RESERVE,
  API_LIST_RESERVES
} from "./urls.js";

describe("reserve api interaction with listing", () => {
  it("should evict cache when creating a reserve", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_RESERVES, {
      response: {
        reserves: [
          {
            reserve_pub: "11",
          } as MerchantBackend.Rewards.ReserveStatusEntry,
        ],
      },
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const api = useReservesAPI();
        const query = useInstanceReserves();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            reserves: [{ reserve_pub: "11" }],
          });

          env.addRequestExpectation(API_CREATE_RESERVE, {
            request: {
              initial_balance: "ARS:3333",
              exchange_url: "http://url",
              wire_method: "iban",
            },
            response: {
              reserve_pub: "22",
              accounts: [],
            },
          });

          env.addRequestExpectation(API_LIST_RESERVES, {
            response: {
              reserves: [
                {
                  reserve_pub: "11",
                } as MerchantBackend.Rewards.ReserveStatusEntry,
                {
                  reserve_pub: "22",
                } as MerchantBackend.Rewards.ReserveStatusEntry,
              ],
            },
          });

          api.createReserve({
            initial_balance: "ARS:3333",
            exchange_url: "http://url",
            wire_method: "iban",
          });
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;

          expect(query.data).deep.equals({
            reserves: [
              {
                reserve_pub: "11",
              } as MerchantBackend.Rewards.ReserveStatusEntry,
              {
                reserve_pub: "22",
              } as MerchantBackend.Rewards.ReserveStatusEntry,
            ],
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });

  it("should evict cache when deleting a reserve", async () => {
    const env = new ApiMockEnvironment();

    env.addRequestExpectation(API_LIST_RESERVES, {
      response: {
        reserves: [
          {
            reserve_pub: "11",
          } as MerchantBackend.Rewards.ReserveStatusEntry,
          {
            reserve_pub: "22",
          } as MerchantBackend.Rewards.ReserveStatusEntry,
          {
            reserve_pub: "33",
          } as MerchantBackend.Rewards.ReserveStatusEntry,
        ],
      },
    });

    const hookBehavior = await tests.hookBehaveLikeThis(
      () => {
        const api = useReservesAPI();
        const query = useInstanceReserves();
        return { query, api };
      },
      {},
      [
        ({ query, api }) => {
          expect(query.loading).true;
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });

          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            reserves: [
              { reserve_pub: "11" },
              { reserve_pub: "22" },
              { reserve_pub: "33" },
            ],
          });

          env.addRequestExpectation(API_DELETE_RESERVE("11"), {});
          env.addRequestExpectation(API_LIST_RESERVES, {
            response: {
              reserves: [
                {
                  reserve_pub: "22",
                } as MerchantBackend.Rewards.ReserveStatusEntry,
                {
                  reserve_pub: "33",
                } as MerchantBackend.Rewards.ReserveStatusEntry,
              ],
            },
          });

          api.deleteReserve("11");
        },
        ({ query, api }) => {
          expect(env.assertJustExpectedRequestWereMade()).deep.eq({
            result: "ok",
          });
          expect(query.loading).false;
          expect(query.ok).true;
          if (!query.ok) return;
          expect(query.data).deep.equals({
            reserves: [{ reserve_pub: "22" }, { reserve_pub: "33" }],
          });
        },
      ],
      env.buildTestingContext(),
    );

    expect(hookBehavior).deep.eq({ result: "ok" });
    expect(env.assertJustExpectedRequestWereMade()).deep.eq({ result: "ok" });
  });
});
