/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  AccessToken,
  AmountJson,
  HttpStatusCode,
  LibtoolVersion,
  OperationFail,
  OperationOk,
  TalerBankConversionApi,
  TalerBankConversionHttpClient,
  TalerCorebankApi,
  TalerCoreBankHttpClient,
  TalerError,
  UserAndToken,
} from "@gnu-taler/taler-util";
import { HttpRequestLibrary } from "@gnu-taler/taler-util/http";
import {
  BrowserHttpLib,
  ErrorLoading,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import {
  ComponentChildren,
  createContext,
  FunctionComponent,
  h,
  VNode,
} from "preact";
import { useContext, useEffect, useState } from "preact/hooks";
import {
  revalidateAccountDetails,
  revalidatePublicAccounts,
  revalidateTransactions,
} from "../hooks/access.js";
import {
  revalidateBusinessAccounts,
  revalidateCashouts,
  revalidateConversionInfo,
} from "../hooks/circuit.js";

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

export type Type = {
  url: URL;
  config: TalerCorebankApi.Config;
  api: TalerCoreBankHttpClient;
  hints: VersionHint[];
};

// FIXME: below
// @ts-expect-error default value to undefined, should it be another thing?
const Context = createContext<Type>(undefined);

export const useBankCoreApiContext = (): Type => useContext(Context);

export enum VersionHint {
  /**
   * when this flag is on, server is running an old version with cashout before implementing 2fa API
   */
  CASHOUT_BEFORE_2FA,
}

export type ConfigResult =
  | undefined
  | { type: "ok"; config: TalerCorebankApi.Config; hints: VersionHint[] }
  | { type: "incompatible"; result: TalerCorebankApi.Config; supported: string }
  | { type: "error"; error: TalerError };

export const BankCoreApiProvider = ({
  baseUrl,
  children,
  frameOnError,
}: {
  baseUrl: string;
  children: ComponentChildren;
  frameOnError: FunctionComponent<{ children: ComponentChildren }>;
}): VNode => {
  const [checked, setChecked] = useState<ConfigResult>();
  const { i18n } = useTranslationContext();
  const url = new URL(baseUrl);
  const api = new CacheAwareTalerCoreBankHttpClient(url.href, new BrowserHttpLib());
  useEffect(() => {
    api
      .getConfig()
      .then((resp) => {
        if (api.isCompatible(resp.body.version)) {
          setChecked({ type: "ok", config: resp.body, hints: [] });
        } else {
          // this API supports version 3.0.3
          const compare = LibtoolVersion.compare("3:0:3", resp.body.version);
          if (compare?.compatible ?? false) {
            setChecked({
              type: "ok",
              config: resp.body,
              hints: [VersionHint.CASHOUT_BEFORE_2FA],
            });
          } else {
            setChecked({
              type: "incompatible",
              result: resp.body,
              supported: api.PROTOCOL_VERSION,
            });
          }
        }
      })
      .catch((error: unknown) => {
        if (error instanceof TalerError) {
          setChecked({ type: "error", error });
        }
      });
  }, []);

  if (checked === undefined) {
    return h(frameOnError, { children: h("div", {}, "loading...") });
  }
  if (checked.type === "error") {
    return h(frameOnError, {
      children: h(ErrorLoading, { error: checked.error, showDetail: true }),
    });
  }
  if (checked.type === "incompatible") {
    return h(frameOnError, {
      children: h(
        "div",
        {},
        i18n.str`The bank backend is not supported. Supported version "${checked.supported}", server version "${checked.result.version}"`,
      ),
    });
  }
  const value: Type = {
    url,
    config: checked.config,
    api: api,
    hints: checked.hints,
  };
  return h(Context.Provider, {
    value,
    children,
  });
};
class CacheAwareTalerBankConversionHttpClient extends TalerBankConversionHttpClient {
  constructor(baseUrl: string, httpClient?: HttpRequestLibrary) {
    super(baseUrl, httpClient);
  }
  async updateConversionRate(auth: AccessToken, body: TalerBankConversionApi.ConversionRate) {
    const resp = await super.updateConversionRate(auth, body);
    if (resp.type === "ok") {
      await revalidateConversionInfo();
    }
    return resp
  }
}

class CacheAwareTalerCoreBankHttpClient extends TalerCoreBankHttpClient {
  constructor(baseUrl: string, httpClient?: HttpRequestLibrary) {
    super(baseUrl, httpClient);
  }
  async deleteAccount(auth: UserAndToken, cid?: string | undefined) {
    const resp = await super.deleteAccount(auth, cid);
    if (resp.type === "ok") {
      await revalidatePublicAccounts();
      await revalidateBusinessAccounts();
    }
    return resp;
  }
  async createAccount(
    auth: AccessToken,
    body: TalerCorebankApi.RegisterAccountRequest,
  ) {
    const resp = await super.createAccount(auth, body);
    if (resp.type === "ok") {
      // admin balance change on new account
      await revalidateAccountDetails();
      await revalidateTransactions();
      await revalidatePublicAccounts();
      await revalidateBusinessAccounts();
    }
    return resp;
  }
  async updateAccount(
    auth: UserAndToken,
    body: TalerCorebankApi.AccountReconfiguration,
    cid?: string | undefined,
  ) {
    const resp = await super.updateAccount(auth, body, cid);
    if (resp.type === "ok") {
      await revalidateAccountDetails();
    }
    return resp;
  }
  async createTransaction(
    auth: UserAndToken,
    body: TalerCorebankApi.CreateTransactionRequest,
    cid?: string | undefined,
  ) {
    const resp = await super.createTransaction(auth, body, cid);
    if (resp.type === "ok") {
      await revalidateAccountDetails();
      await revalidateTransactions();
    }
    return resp;
  }
  async confirmWithdrawalById(
    auth: UserAndToken,
    wid: string,
    cid?: string | undefined,
  ) {
    const resp = await super.confirmWithdrawalById(auth, wid, cid);
    if (resp.type === "ok") {
      await revalidateAccountDetails();
      await revalidateTransactions();
    }
    return resp;
  }
  async createCashout(
    auth: UserAndToken,
    body: TalerCorebankApi.CashoutRequest,
    cid?: string | undefined,
  ) {
    const resp = await super.createCashout(auth, body, cid);
    if (resp.type === "ok") {
      await revalidateAccountDetails();
      await revalidateCashouts();
      await revalidateTransactions();
    }
    return resp;
  }

  getConversionInfoAPI(): TalerBankConversionHttpClient {
    const api = super.getConversionInfoAPI();
    return new CacheAwareTalerBankConversionHttpClient(api.baseUrl, this.httpLib)
  }
}

export const BankCoreApiProviderTesting = ({
  children,
  state,
  url,
}: {
  children: ComponentChildren;
  state: TalerCorebankApi.Config;
  url: string;
}): VNode => {
  const value: Type = {
    url: new URL(url),
    config: state,
    // @ts-expect-error this API is not being used, not really needed
    api: undefined,
    hints: [],
  };

  return h(Context.Provider, {
    value,
    children,
  });
};
