/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  Codec,
  TranslatedString,
  buildCodecForObject,
  codecForBoolean,
  codecForNumber,
  codecForString,
  codecOptional
} from "@gnu-taler/taler-util";
import { buildStorageKey, useLocalStorage, useTranslationContext } from "@gnu-taler/web-util/browser";

interface Settings {
  allowInsecurePassword: boolean;
  keepSessionAfterReload: boolean;
}

export function getAllBooleanSettings(): Array<keyof Settings> {
  return ["allowInsecurePassword", "keepSessionAfterReload"]
}

export function getLabelForSetting(k: keyof Settings, i18n: ReturnType<typeof useTranslationContext>["i18n"]): TranslatedString {
  switch (k) {
    case "allowInsecurePassword": return i18n.str`Allow Insecure password`
    case "keepSessionAfterReload": return i18n.str`Keep session after reload`
  }
}

export const codecForSettings = (): Codec<Settings> =>
  buildCodecForObject<Settings>()
  .property("allowInsecurePassword", (codecForBoolean()))
  .property("keepSessionAfterReload", (codecForBoolean()))
  .build("Settings");

const defaultSettings: Settings = {
  allowInsecurePassword: false,
  keepSessionAfterReload: false,
};

const EXCHANGE_SETTINGS_KEY = buildStorageKey(
  "exchange-settings",
  codecForSettings(),
);

export function useSettings(): [
  Readonly<Settings>,
  <T extends keyof Settings>(key: T, value: Settings[T]) => void,
] {
  const { value, update } = useLocalStorage(
    EXCHANGE_SETTINGS_KEY,
    defaultSettings,
  );

  function updateField<T extends keyof Settings>(k: T, v: Settings[T]) {
    const newValue = { ...value, [k]: v };
    update(newValue);
  }
  return [value, updateField];
}
