/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  ExchangeListItem,
  ExchangeTosStatus,
  LibtoolVersion,
  TranslatedString,
  WalletCoreVersion,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { Pages } from "../NavigationBar.js";
import { Checkbox } from "../components/Checkbox.js";
import { EnabledBySettings } from "../components/EnabledBySettings.js";
import { Part } from "../components/Part.js";
import { SelectList } from "../components/SelectList.js";
import {
  DestructiveText,
  Input,
  LinkPrimary,
  SubTitle,
  SuccessText,
  WarningBox,
  WarningText,
} from "../components/styled/index.js";
import { useAlertContext } from "../context/alert.js";
import { useBackendContext } from "../context/backend.js";
import { useAsyncAsHook } from "../hooks/useAsyncAsHook.js";
import { useBackupDeviceName } from "../hooks/useBackupDeviceName.js";
import { useSettings } from "../hooks/useSettings.js";
import { ToggleHandler } from "../mui/handlers.js";
import { Settings } from "../platform/api.js";
import { platform } from "../platform/foreground.js";
import { WALLET_CORE_SUPPORTED_VERSION } from "../wxApi.js";

const GIT_HASH = typeof __GIT_HASH__ !== "undefined" ? __GIT_HASH__ : undefined;

export function SettingsPage(): VNode {
  const [settings, updateSettings] = useSettings();
  const { safely } = useAlertContext();
  const { name, update } = useBackupDeviceName();
  const webex = platform.getWalletWebExVersion();
  const api = useBackendContext();

  const exchangesHook = useAsyncAsHook(async () => {
    const list = await api.wallet.call(WalletApiOperation.ListExchanges, {});
    const version = await api.wallet.call(WalletApiOperation.GetVersion, {});
    return { exchanges: list.exchanges, version };
  });
  const { exchanges, version } =
    !exchangesHook || exchangesHook.hasError
      ? { exchanges: [], version: undefined }
      : exchangesHook.response;

  return (
    <SettingsView
      knownExchanges={exchanges}
      deviceName={name}
      setDeviceName={update}
      autoOpenToggle={{
        value: settings.autoOpen,
        button: {
          onClick: safely("update support injection", async () => {
            updateSettings("autoOpen", !settings.autoOpen);
          }),
        },
      }}
      injectTalerToggle={{
        value: settings.injectTalerSupport,
        button: {
          onClick: safely("update support injection", async () => {
            updateSettings("injectTalerSupport", !settings.injectTalerSupport);
          }),
        },
      }}
      advanceToggle={{
        value: settings.advanceMode,
        button: {
          onClick: safely("update advance mode", async () => {
            updateSettings("advanceMode", !settings.advanceMode);
          }),
        },
      }}
      langToggle={{
        value: settings.langSelector,
        button: {
          onClick: safely("update lang selector", async () => {
            updateSettings("langSelector", !settings.langSelector);
          }),
        },
      }}
      webexVersion={{
        version: webex.version,
        hash: GIT_HASH,
      }}
      coreVersion={version}
    />
  );
}

export interface ViewProps {
  deviceName: string;
  setDeviceName: (s: string) => Promise<void>;
  autoOpenToggle: ToggleHandler;
  injectTalerToggle: ToggleHandler;
  advanceToggle: ToggleHandler;
  langToggle: ToggleHandler;
  knownExchanges: Array<ExchangeListItem>;
  coreVersion: WalletCoreVersion | undefined;
  webexVersion: {
    version: string;
    hash: string | undefined;
  };
}

export function SettingsView({
  knownExchanges,
  autoOpenToggle,
  injectTalerToggle,
  advanceToggle,
  langToggle,
  coreVersion,
  webexVersion,
}: ViewProps): VNode {
  const { i18n, lang, supportedLang, changeLanguage } = useTranslationContext();

  return (
    <Fragment>
      <section>
        <SubTitle>
          <i18n.Translate>Trust</i18n.Translate>
        </SubTitle>
        {!knownExchanges || !knownExchanges.length ? (
          <div>
            <i18n.Translate>No exchange yet</i18n.Translate>
          </div>
        ) : (
          <Fragment>
            <table>
              <thead>
                <tr>
                  <th>
                    <i18n.Translate>Currency</i18n.Translate>
                  </th>
                  <th>
                    <i18n.Translate>URL</i18n.Translate>
                  </th>
                  <th>
                    <i18n.Translate>Term of Service</i18n.Translate>
                  </th>
                </tr>
              </thead>
              <tbody>
                {knownExchanges.map((e, idx) => {
                  function Status(): VNode {
                    switch (e.tosStatus) {
                      case ExchangeTosStatus.Accepted:
                        return (
                          <SuccessText>
                            <i18n.Translate>ok</i18n.Translate>
                          </SuccessText>
                        );
                      case ExchangeTosStatus.Pending:
                        return (
                          <WarningText>
                            <i18n.Translate>pending</i18n.Translate>
                          </WarningText>
                        );
                      case ExchangeTosStatus.Proposed:
                        return (
                          <i18n.Translate>proposed</i18n.Translate>
                        );
                      default:
                        return (
                          <DestructiveText>
                            <i18n.Translate>
                              unknown (exchange status should be updated)
                            </i18n.Translate>
                          </DestructiveText>
                        );
                    }
                  }
                  return (
                    <tr key={idx}>
                      <td>{e.currency}</td>
                      <td>
                        <a href={e.exchangeBaseUrl}>{e.exchangeBaseUrl}</a>
                      </td>
                      <td>
                        <Status />
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
          </Fragment>
        )}
        <div style={{ display: "flex", justifyContent: "space-between" }}>
          <div />
          <LinkPrimary href={Pages.settingsExchangeAdd({})}>
            <i18n.Translate>Add an exchange</i18n.Translate>
          </LinkPrimary>
        </div>

        {coreVersion && (<Fragment>
          {LibtoolVersion.compare(coreVersion.version, WALLET_CORE_SUPPORTED_VERSION)?.compatible ? undefined :
            <WarningBox>
              <i18n.Translate>
                The version of wallet core is not supported. (supported version: {WALLET_CORE_SUPPORTED_VERSION})
              </i18n.Translate>
            </WarningBox>}
          <EnabledBySettings name="advanceMode">
            <Part
              title={i18n.str`Exchange compatibility`}
              text={<span>{coreVersion.exchange}</span>}
            />
            <Part
              title={i18n.str`Merchant compatibility`}
              text={<span>{coreVersion.merchant}</span>}
            />
            <Part
              title={i18n.str`Bank compatibility`}
              text={<span>{coreVersion.bank}</span>}
            />
            <Part
              title={i18n.str`Wallet Core compatibility`}
              text={<span>{coreVersion.version}</span>}
            />
          </EnabledBySettings>
        </Fragment>
        )}
        <SubTitle>
          <i18n.Translate>Advance mode</i18n.Translate>
        </SubTitle>
        <Checkbox
          label={i18n.str`Enable advance mode`}
          name="devMode"
          description={i18n.str`Show more information and options in the UI`}
          enabled={advanceToggle.value!}
          onToggle={advanceToggle.button.onClick!}
        />
        <EnabledBySettings name="advanceMode">
          <AdvanceSettings />
        </EnabledBySettings>
        <EnabledBySettings name="langSelector">
          <SubTitle>
            <i18n.Translate>Display</i18n.Translate>
          </SubTitle>
          <Input>
            <SelectList
              label={<i18n.Translate>Current Language</i18n.Translate>}
              list={supportedLang}
              name="lang"
              value={lang}
              onChange={(v) => changeLanguage(v)}
            />
          </Input>
        </EnabledBySettings>
        <SubTitle>
          <i18n.Translate>Navigator</i18n.Translate>
        </SubTitle>
        <Checkbox
          label={i18n.str`Inject Taler support in all pages`}
          name="inject"
          description={
            <i18n.Translate>
              Disabling this option will make some web application not able to
              trigger the wallet when clicking links but you will be able to
              open the wallet using the keyboard shortcut
            </i18n.Translate>
          }
          enabled={injectTalerToggle.value!}
          onToggle={injectTalerToggle.button.onClick!}
        />
        <Checkbox
          label={i18n.str`Automatically open wallet`}
          name="autoOpen"
          description={
            <i18n.Translate>
              Open the wallet when a payment action is found.
            </i18n.Translate>
          }
          enabled={autoOpenToggle.value!}
          onToggle={autoOpenToggle.button.onClick!}
        />
        <SubTitle>
          <i18n.Translate>Version</i18n.Translate>
        </SubTitle>
        <Part
          title={i18n.str`Web Extension`}
          text={
            <span>
              {webexVersion.version}{" "}
              <EnabledBySettings name="advanceMode">
                {webexVersion.hash}
              </EnabledBySettings>
            </span>
          }
        />
      </section>
    </Fragment>
  );
}

type Info = { label: TranslatedString; description: TranslatedString };
type Options = {
  [k in keyof Settings]?: Info;
};
function AdvanceSettings(): VNode {
  const [settings, updateSettings] = useSettings();
  const { i18n } = useTranslationContext();
  const o: Options = {
    backup: {
      label: i18n.str`Show backup feature`,
      description: i18n.str`Backup integration still in beta.`,
    },
    suspendIndividualTransaction: {
      label: i18n.str`Show suspend/resume transaction`,
      description: i18n.str`Prevent transaction from doing network request.`,
    },
    extendedAccountTypes: {
      label: i18n.str`Show more account types on deposit`,
      description: i18n.str`Extends the UI to more payment target types.`,
    },
    showJsonOnError: {
      label: i18n.str`Show JSON on error`,
      description: i18n.str`Print more information about the error. Useful for debugging.`,
    },
    walletAllowHttp: {
      label: i18n.str`Allow HTTP connections`,
      description: i18n.str`Using HTTP connection may be faster but unsafe (wallet restart required)`,
    },
    langSelector: {
      label: i18n.str`Lang selector`,
      description: i18n.str`Allows to manually change the language of the UI. Otherwise it will be automatically selected by your browser configuration.`,
    },
  };
  return (
    <Fragment>
      <section>
        {Object.entries(o).map(([name, { label, description }]) => {
          const settingsName = name as keyof Settings;
          return (
            <Checkbox
              label={label}
              name={name}
              description={description}
              enabled={settings[settingsName]}
              onToggle={async () => {
                updateSettings(settingsName, !settings[settingsName]);
              }}
            />
          );
        })}
      </section>
    </Fragment>
  );
}
