/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  AmountJson,
  AmountString,
  ExchangeListItem,
} from "@gnu-taler/taler-util";
import { Loading } from "../../components/Loading.js";
import { State as SelectExchangeState } from "../../hooks/useSelectedExchange.js";
import {
  AmountFieldHandler,
  ButtonHandler,
  SelectFieldHandler,
} from "../../mui/handlers.js";
import { StateViewMap, compose } from "../../utils/index.js";
import {
  useComponentStateFromParams,
  useComponentStateFromURI,
} from "./state.js";

import { ErrorAlertView } from "../../components/CurrentAlerts.js";
import { ErrorAlert } from "../../context/alert.js";
import { ExchangeSelectionPage } from "../../wallet/ExchangeSelection/index.js";
import { NoExchangesView } from "../../wallet/ExchangeSelection/views.js";
import { SelectAmountView, SuccessView } from "./views.js";

export interface PropsFromURI {
  talerWithdrawUri: string | undefined;
  cancel: () => Promise<void>;
  onSuccess: (txid: string) => Promise<void>;
}

export interface PropsFromParams {
  talerExchangeWithdrawUri: string | undefined;
  amount: string | undefined;
  cancel: () => Promise<void>;
  onSuccess: (txid: string) => Promise<void>;
  onAmountChanged: (amount: AmountString) => Promise<void>;
}

export type State =
  | State.Loading
  | State.LoadingUriError
  | SelectExchangeState.NoExchangeFound
  | SelectExchangeState.Selecting
  | State.SelectAmount
  | State.Success;

export namespace State {
  export interface Loading {
    status: "loading";
    error: undefined;
  }
  export interface LoadingUriError {
    status: "error";
    error: ErrorAlert;
  }

  export interface SelectAmount {
    status: "select-amount";
    error: undefined;
    exchangeBaseUrl: string;
    confirm: ButtonHandler;
    amount: AmountFieldHandler;
    currency: string;
  }

  export type Success = {
    status: "success";
    error: undefined;

    currentExchange: ExchangeListItem;

    chosenAmount: AmountJson;
    withdrawalFee: AmountJson;
    toBeReceived: AmountJson;

    doWithdrawal: ButtonHandler;
    doSelectExchange: ButtonHandler;

    ageRestriction?: SelectFieldHandler;

    talerWithdrawUri?: string;
    cancel: () => Promise<void>;
    onTosUpdate: () => void;
  };
}

const viewMapping: StateViewMap<State> = {
  loading: Loading,
  error: ErrorAlertView,
  "select-amount": SelectAmountView,
  "no-exchange-found": NoExchangesView,
  "selecting-exchange": ExchangeSelectionPage,
  success: SuccessView,
};

export const WithdrawPageFromURI = compose(
  "WithdrawPageFromURI_Withdraw",
  (p: PropsFromURI) => useComponentStateFromURI(p),
  viewMapping,
);
export const WithdrawPageFromParams = compose(
  "WithdrawPageFromParams",
  (p: PropsFromParams) => useComponentStateFromParams(p),
  viewMapping,
);
