/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { Amounts } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { alertFromError, useAlertContext } from "../../context/alert.js";
import { useBackendContext } from "../../context/backend.js";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { useAsyncAsHook } from "../../hooks/useAsyncAsHook.js";
import { Props, State } from "./index.js";

export function useComponentState({
  talerTipUri: talerRewardUri,
  onCancel,
  onSuccess,
}: Props): State {
  const api = useBackendContext();
  const { i18n } = useTranslationContext();
  const { pushAlertOnError } = useAlertContext();
  const tipInfo = useAsyncAsHook(async () => {
    if (!talerRewardUri) throw Error("ERROR_NO-URI-FOR-TIP");
    const tip = await api.wallet.call(WalletApiOperation.PrepareReward, {
      talerRewardUri,
    });
    return { tip };
  });

  if (!tipInfo) {
    return {
      status: "loading",
      error: undefined,
    };
  }
  if (tipInfo.hasError) {
    return {
      status: "error",
      error: alertFromError(
        i18n.str`Could not load the status of the term of service`,
        tipInfo,
      ),
    };
  }
  // if (tipInfo.hasError) {
  //   return {
  //     status: "loading-uri",
  //     error: tipInfo,
  //   };
  // }

  const { tip } = tipInfo.response;

  const doAccept = async (): Promise<void> => {
    const res = await api.wallet.call(WalletApiOperation.AcceptReward, {
      walletRewardId: tip.walletRewardId,
    });

    //FIX: this may not be seen since we are moving to the success also
    tipInfo.retry();
    onSuccess(res.transactionId);
  };

  const baseInfo = {
    merchantBaseUrl: tip.merchantBaseUrl,
    exchangeBaseUrl: tip.exchangeBaseUrl,
    amount: Amounts.parseOrThrow(tip.rewardAmountEffective),
    error: undefined,
    cancel: {
      onClick: pushAlertOnError(onCancel),
    },
  };

  if (tip.accepted) {
    return {
      status: "accepted",
      ...baseInfo,
    };
  }

  return {
    status: "ready",
    ...baseInfo,
    accept: {
      onClick: pushAlertOnError(doAccept),
    },
  };
}
