/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  Amounts,
  NotificationType,
  TransactionPayment,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { useEffect, useState } from "preact/hooks";
import { alertFromError, useAlertContext } from "../../context/alert.js";
import { useBackendContext } from "../../context/backend.js";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { useAsyncAsHook } from "../../hooks/useAsyncAsHook.js";
import { Props, State } from "./index.js";

export function useComponentState({
  talerRefundUri,
  cancel,
  onSuccess,
}: Props): State {
  const api = useBackendContext();
  const { i18n } = useTranslationContext();
  const [ignored, setIgnored] = useState(false);
  const { pushAlertOnError } = useAlertContext();

  const info = useAsyncAsHook(async () => {
    if (!talerRefundUri) throw Error("ERROR_NO-URI-FOR-REFUND");
    const refund = await api.wallet.call(
      WalletApiOperation.StartRefundQueryForUri,
      {
        talerRefundUri,
      },
    );
    const purchase = await api.wallet.call(
      WalletApiOperation.GetTransactionById,
      {
        transactionId: refund.transactionId,
      },
    );
    if (purchase.type !== TransactionType.Payment) {
      throw Error("Refund of non purchase transaction is not handled");
    }
    return { refund, purchase, uri: talerRefundUri };
  });

  useEffect(() =>
    api.listener.onUpdateNotification(
      [NotificationType.TransactionStateTransition],
      info?.retry,
    ),
  );

  if (!info) {
    return { status: "loading", error: undefined };
  }
  if (info.hasError) {
    return {
      status: "error",
      error: alertFromError(
        i18n.str`Could not load the status of the term of service`,
        info,
      ),
    };
  }
  // if (info.hasError) {
  //   return {
  //     status: "loading-uri",
  //     error: info,
  //   };
  // }

  const { refund, purchase, uri } = info.response;

  const doAccept = async (): Promise<void> => {
    const res = await api.wallet.call(
      WalletApiOperation.StartRefundQueryForUri,
      {
        talerRefundUri: uri,
      },
    );

    onSuccess(res.transactionId);
  };

  const doIgnore = async (): Promise<void> => {
    setIgnored(true);
  };

  const baseInfo = {
    amount: Amounts.parseOrThrow(purchase.amountEffective),
    // granted: Amounts.parseOrThrow(info.response.refund.granted),
    // awaitingAmount: Amounts.parseOrThrow(refund.awaiting),
    merchantName: purchase.info.merchant.name,
    // products: purchase.info.products,
    error: undefined,
  };

  if (ignored) {
    return {
      status: "ignored",
      ...baseInfo,
    };
  }

  //FIXME: DD37 wallet-core is not returning this value
  // if (refund.pending) {
  //   return {
  //     status: "in-progress",
  //     ...baseInfo,
  //   };
  // }

  return {
    status: "ready",
    ...baseInfo,
    orderId: purchase.info.orderId,
    accept: {
      onClick: pushAlertOnError(doAccept),
    },
    ignore: {
      onClick: pushAlertOnError(doIgnore),
    },
    cancel,
  };
}
