/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  AbsoluteTime,
  Amounts,
  MerchantContractTerms as ContractTerms,
  PreparePayResultType,
  TranslatedString,
} from "@gnu-taler/taler-util";
import { Fragment, h, VNode } from "preact";
import { Part } from "../../components/Part.js";
import { PaymentButtons } from "../../components/PaymentButtons.js";
import { SuccessBox, WarningBox } from "../../components/styled/index.js";
import { Time } from "../../components/Time.js";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import {
  getAmountWithFee,
  MerchantDetails,
  PurchaseDetails,
} from "../../wallet/Transaction.js";
import { State } from "./index.js";

type SupportedStates =
  | State.Ready
  | State.Confirmed
  | State.NoBalanceForCurrency
  | State.NoEnoughBalance;

export function BaseView(state: SupportedStates): VNode {
  const { i18n } = useTranslationContext();

  const contractTerms: ContractTerms = state.payStatus.contractTerms;

  const effective =
    "amountEffective" in state.payStatus
      ? state.payStatus.amountEffective
        ? Amounts.parseOrThrow(state.payStatus.amountEffective)
        : Amounts.zeroOfCurrency(state.amount.currency)
      : state.amount;

  return (
    <Fragment>
      <ShowImportantMessage state={state} />

      <section style={{ textAlign: "left" }}>
        <Part
          title={i18n.str`Purchase`}
          text={contractTerms.summary as TranslatedString}
          kind="neutral"
        />
        <Part
          title={i18n.str`Merchant`}
          text={<MerchantDetails merchant={contractTerms.merchant} />}
          kind="neutral"
        />
        <Part
          title={i18n.str`Details`}
          text={
            <PurchaseDetails
              price={getAmountWithFee(effective, state.amount, "debit")}
              info={{
                ...contractTerms,
                orderId: contractTerms.order_id,
                contractTermsHash: "",
                // products: contractTerms.products!,
              }}
              proposalId={state.payStatus.proposalId}
            />
          }
          kind="neutral"
        />
        {contractTerms.order_id && (
          <Part
            title={i18n.str`Receipt`}
            text={`#${contractTerms.order_id}` as TranslatedString}
            kind="neutral"
          />
        )}
        {contractTerms.pay_deadline && (
          <Part
            title={i18n.str`Valid until`}
            text={
              <Time
                timestamp={AbsoluteTime.fromProtocolTimestamp(
                  contractTerms.pay_deadline,
                )}
                format="dd MMMM yyyy, HH:mm"
              />
            }
            kind="neutral"
          />
        )}
      </section>
      <PaymentButtons
        amount={effective}
        payStatus={state.payStatus}
        uri={state.uri}
        payHandler={state.status === "ready" ? state.payHandler : undefined}
        goToWalletManualWithdraw={state.goToWalletManualWithdraw}
      />
    </Fragment>
  );
}

function ShowImportantMessage({ state }: { state: SupportedStates }): VNode {
  const { i18n } = useTranslationContext();
  const { payStatus } = state;
  if (payStatus.status === PreparePayResultType.AlreadyConfirmed) {
    if (payStatus.paid) {
      if (payStatus.contractTerms.fulfillment_url) {
        return (
          <SuccessBox>
            <i18n.Translate>
              Already paid, you are going to be redirected to{" "}
              <a href={payStatus.contractTerms.fulfillment_url}>
                {payStatus.contractTerms.fulfillment_url}
              </a>
            </i18n.Translate>
          </SuccessBox>
        );
      }
      return (
        <SuccessBox>
          <i18n.Translate>Already paid</i18n.Translate>
        </SuccessBox>
      );
    }
    return (
      <WarningBox>
        <i18n.Translate>Already claimed</i18n.Translate>
      </WarningBox>
    );
  }

  return <Fragment />;
}
