/*
 This file is part of GNU Taler
 (C) 2019 GNUnet e.V.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import test from "ava";
import {
  parsePayPushUri,
  parsePayTemplateUri,
  parsePayUri,
  parseRefundUri,
  parseRestoreUri,
  parseRewardUri,
  parseWithdrawExchangeUri,
  parseWithdrawUri,
  stringifyPayPushUri,
  stringifyPayUri,
  stringifyRestoreUri,
  stringifyWithdrawExchange,
} from "./taleruri.js";
import { AmountString } from "./taler-types.js";

test("taler pay url parsing: wrong scheme", (t) => {
  const url1 = "talerfoo://";
  const r1 = parsePayUri(url1);
  t.is(r1, undefined);

  const url2 = "taler://refund/a/b/c/d/e/f";
  const r2 = parsePayUri(url2);
  t.is(r2, undefined);
});

test("taler pay url parsing: defaults", (t) => {
  const url1 = "taler://pay/example.com/myorder/";
  const r1 = parsePayUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.merchantBaseUrl, "https://example.com/");
  t.is(r1.sessionId, "");

  const url2 = "taler://pay/example.com/myorder/mysession";
  const r2 = parsePayUri(url2);
  if (!r2) {
    t.fail();
    return;
  }
  t.is(r2.merchantBaseUrl, "https://example.com/");
  t.is(r2.sessionId, "mysession");
});

test("taler pay url parsing: instance", (t) => {
  const url1 = "taler://pay/example.com/instances/myinst/myorder/";
  const r1 = parsePayUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.merchantBaseUrl, "https://example.com/instances/myinst/");
  t.is(r1.orderId, "myorder");
});

test("taler pay url parsing (claim token)", (t) => {
  const url1 = "taler://pay/example.com/instances/myinst/myorder/?c=ASDF";
  const r1 = parsePayUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.merchantBaseUrl, "https://example.com/instances/myinst/");
  t.is(r1.orderId, "myorder");
  t.is(r1.claimToken, "ASDF");
});

test("taler refund uri parsing: non-https #1", (t) => {
  const url1 = "taler+http://refund/example.com/myorder/";
  const r1 = parseRefundUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.merchantBaseUrl, "http://example.com/");
  t.is(r1.orderId, "myorder");
});

test("taler pay uri parsing: non-https", (t) => {
  const url1 = "taler+http://pay/example.com/myorder/";
  const r1 = parsePayUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.merchantBaseUrl, "http://example.com/");
  t.is(r1.orderId, "myorder");
});

test("taler pay uri parsing: missing session component", (t) => {
  const url1 = "taler+http://pay/example.com/myorder";
  const r1 = parsePayUri(url1);
  if (r1) {
    t.fail();
    return;
  }
  t.pass();
});

test("taler withdraw uri parsing", (t) => {
  const url1 = "taler://withdraw/bank.example.com/12345";
  const r1 = parseWithdrawUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.withdrawalOperationId, "12345");
  t.is(r1.bankIntegrationApiBaseUrl, "https://bank.example.com/");
});

test("taler withdraw uri parsing (http)", (t) => {
  const url1 = "taler+http://withdraw/bank.example.com/12345";
  const r1 = parseWithdrawUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.withdrawalOperationId, "12345");
  t.is(r1.bankIntegrationApiBaseUrl, "http://bank.example.com/");
});

test("taler refund uri parsing", (t) => {
  const url1 = "taler://refund/merchant.example.com/1234/";
  const r1 = parseRefundUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.merchantBaseUrl, "https://merchant.example.com/");
  t.is(r1.orderId, "1234");
});

test("taler refund uri parsing with instance", (t) => {
  const url1 = "taler://refund/merchant.example.com/instances/myinst/1234/";
  const r1 = parseRefundUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.orderId, "1234");
  t.is(r1.merchantBaseUrl, "https://merchant.example.com/instances/myinst/");
});

test("taler reward pickup uri", (t) => {
  const url1 = "taler://reward/merchant.example.com/tipid";
  const r1 = parseRewardUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.merchantBaseUrl, "https://merchant.example.com/");
});

test("taler reward pickup uri with instance", (t) => {
  const url1 = "taler://reward/merchant.example.com/instances/tipm/tipid";
  const r1 = parseRewardUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.merchantBaseUrl, "https://merchant.example.com/instances/tipm/");
  t.is(r1.merchantRewardId, "tipid");
});

test("taler reward pickup uri with instance and prefix", (t) => {
  const url1 = "taler://reward/merchant.example.com/my/pfx/tipm/tipid";
  const r1 = parseRewardUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.merchantBaseUrl, "https://merchant.example.com/my/pfx/tipm/");
  t.is(r1.merchantRewardId, "tipid");
});

test("taler peer to peer push URI", (t) => {
  const url1 = "taler://pay-push/exch.example.com/foo";
  const r1 = parsePayPushUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.exchangeBaseUrl, "https://exch.example.com/");
  t.is(r1.contractPriv, "foo");
});

test("taler peer to peer push URI (path)", (t) => {
  const url1 = "taler://pay-push/exch.example.com:123/bla/foo";
  const r1 = parsePayPushUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.exchangeBaseUrl, "https://exch.example.com:123/bla/");
  t.is(r1.contractPriv, "foo");
});

test("taler peer to peer push URI (http)", (t) => {
  const url1 = "taler+http://pay-push/exch.example.com:123/bla/foo";
  const r1 = parsePayPushUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.is(r1.exchangeBaseUrl, "http://exch.example.com:123/bla/");
  t.is(r1.contractPriv, "foo");
});

test("taler peer to peer push URI (stringify)", (t) => {
  const url = stringifyPayPushUri({
    exchangeBaseUrl: "https://foo.example.com/bla/",
    contractPriv: "123",
  });
  t.deepEqual(url, "taler://pay-push/foo.example.com/bla/123");
});
test("taler pay URI (stringify)", (t) => {
  const url1 = stringifyPayUri({
    merchantBaseUrl: "http://localhost:123/",
    orderId: "foo",
    sessionId: "",
  });
  t.deepEqual(url1, "taler+http://pay/localhost:123/foo/");

  const url2 = stringifyPayUri({
    merchantBaseUrl: "http://localhost:123/",
    orderId: "foo",
    sessionId: "bla",
  });
  t.deepEqual(url2, "taler+http://pay/localhost:123/foo/bla");
});

test("taler pay URI (stringify with https)", (t) => {
  const url1 = stringifyPayUri({
    merchantBaseUrl: "https://localhost:123/",
    orderId: "foo",
    sessionId: "",
  });
  t.deepEqual(url1, "taler://pay/localhost:123/foo/");

  const url2 = stringifyPayUri({
    merchantBaseUrl: "https://localhost/",
    orderId: "foo",
    sessionId: "bla",
    noncePriv: "123",
  });
  t.deepEqual(url2, "taler://pay/localhost/foo/bla?n=123");
});

test("taler pay template URI (parsing)", (t) => {
  const url1 =
    "taler://pay-template/merchant.example.com/FEGHYJY48FEGU6WETYIOIDEDE2QW3OCZVY?amount=KUDOS:5";
  const r1 = parsePayTemplateUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.deepEqual(r1.merchantBaseUrl, "https://merchant.example.com/");
  t.deepEqual(r1.templateId, "FEGHYJY48FEGU6WETYIOIDEDE2QW3OCZVY");
  t.deepEqual(r1.templateParams.amount, "KUDOS:5");
});

test("taler pay template URI (parsing, http with port)", (t) => {
  const url1 =
    "taler+http://pay-template/merchant.example.com:1234/FEGHYJY48FEGU6WETYIOIDEDE2QW3OCZVY?amount=KUDOS:5";
  const r1 = parsePayTemplateUri(url1);
  if (!r1) {
    t.fail();
    return;
  }
  t.deepEqual(r1.merchantBaseUrl, "http://merchant.example.com:1234/");
  t.deepEqual(r1.templateId, "FEGHYJY48FEGU6WETYIOIDEDE2QW3OCZVY");
  t.deepEqual(r1.templateParams.amount, "KUDOS:5");
});

test("taler restore URI (parsing, http with port)", (t) => {
  const r1 = parseRestoreUri(
    "taler+http://restore/GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0/prov1.example.com,prov2.example.com:123",
  );
  if (!r1) {
    t.fail();
    return;
  }
  t.deepEqual(
    r1.walletRootPriv,
    "GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0",
  );
  t.deepEqual(r1.providers[0], "http://prov1.example.com/");
  t.deepEqual(r1.providers[1], "http://prov2.example.com:123/");
});
test("taler restore URI (parsing, https with port)", (t) => {
  const r1 = parseRestoreUri(
    "taler://restore/GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0/prov1.example.com,prov2.example.com:234,https%3A%2F%2Fprov1.example.com%2F",
  );
  if (!r1) {
    t.fail();
    return;
  }
  t.deepEqual(
    r1.walletRootPriv,
    "GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0",
  );
  t.deepEqual(r1.providers[0], "https://prov1.example.com/");
  t.deepEqual(r1.providers[1], "https://prov2.example.com:234/");
});

test("taler restore URI (stringify)", (t) => {
  const url = stringifyRestoreUri({
    walletRootPriv: "GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0",
    providers: ["http://prov1.example.com", "https://prov2.example.com:234/"],
  });
  t.deepEqual(
    url,
    "taler://restore/GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0/http%3A%2F%2Fprov1.example.com%2F,https%3A%2F%2Fprov2.example.com%3A234%2F",
  );
});

test("taler withdraw exchange URI (parse)", (t) => {
  const r1 = parseWithdrawExchangeUri(
    "taler://withdraw-exchange/exchange.demo.taler.net/someroot/GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0?a=KUDOS%3A2",
  );
  if (!r1) {
    t.fail();
    return;
  }
  t.deepEqual(
    r1.exchangePub,
    "GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0",
  );
  t.deepEqual(r1.exchangeBaseUrl, "https://exchange.demo.taler.net/someroot/");
  t.deepEqual(r1.amount, "KUDOS:2");
});

test("taler withdraw exchange URI (stringify)", (t) => {
  const url = stringifyWithdrawExchange({
    exchangeBaseUrl: "https://exchange.demo.taler.net",
    exchangePub: "GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0",
  });
  t.deepEqual(
    url,
    "taler://withdraw-exchange/exchange.demo.taler.net/GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0",
  );
});

test("taler withdraw exchange URI with amount (stringify)", (t) => {
  const url = stringifyWithdrawExchange({
    exchangeBaseUrl: "https://exchange.demo.taler.net",
    exchangePub: "GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0",
    amount: "KUDOS:19" as AmountString,
  });
  t.deepEqual(
    url,
    "taler://withdraw-exchange/exchange.demo.taler.net/GJKG23V4ZBHEH45YRK7TWQE8ZTY7JWTY5094TQJSRZN5DSDBX8E0?a=KUDOS%3A19",
  );
});
