import {
  LockedAccount,
  OfficerAccount,
  OfficerId,
  SigningKey,
  createEddsaKeyPair,
  decodeCrock,
  decryptWithDerivedKey,
  eddsaGetPublic,
  encodeCrock,
  encryptWithDerivedKey,
  getRandomBytesF,
  stringToBytes
} from "@gnu-taler/taler-util";

/**
 * Restore previous session and unlock account with password
 *
 * @param salt string from which crypto params will be derived
 * @param key secured private key
 * @param password password for the private key
 * @returns
 */
export async function unlockOfficerAccount(
  account: LockedAccount,
  password: string,
): Promise<OfficerAccount> {
  const rawKey = decodeCrock(account);
  const rawPassword = stringToBytes(password);

  const signingKey = (await decryptWithDerivedKey(
    rawKey,
    rawPassword,
    password,
  ).catch((e: Error) => {
    throw new UnwrapKeyError(e.message);
  })) as SigningKey;

  const publicKey = eddsaGetPublic(signingKey);

  const accountId = encodeCrock(publicKey) as OfficerId;

  return { id: accountId, signingKey };
}

/**
 * Create new account (secured private key)
 * secured with the given password
 *
 * @param sessionId
 * @param password
 * @returns
 */
export async function createNewOfficerAccount(
  password: string,
): Promise<OfficerAccount & { safe: LockedAccount }> {
  const { eddsaPriv, eddsaPub } = createEddsaKeyPair();

  const key = stringToBytes(password);

  const protectedPrivKey = await encryptWithDerivedKey(
    getRandomBytesF(24),
    key,
    eddsaPriv,
    password,
  );

  const signingKey = eddsaPriv as SigningKey;
  const accountId = encodeCrock(eddsaPub) as OfficerId;
  const safe = encodeCrock(protectedPrivKey) as LockedAccount;

  return { id: accountId, signingKey, safe };
}

export class UnwrapKeyError extends Error {
  public cause: string;
  constructor(cause: string) {
    super(`Recovering private key failed on: ${cause}`);
    this.cause = cause;
  }
}
