import { HttpRequestLibrary } from "../http-common.js";
import { HttpStatusCode } from "../http-status-codes.js";
import { createPlatformHttpLib } from "../http.js";
import { LibtoolVersion } from "../libtool-version.js";
import { FailCasesByMethod, ResultByMethod, opSuccess, opUnknownFailure } from "../operation.js";
import { codecForMerchantConfig } from "./types.js";

export type TalerMerchantResultByMethod<prop extends keyof TalerMerchantHttpClient> = ResultByMethod<TalerMerchantHttpClient, prop>
export type TalerMerchantErrorsByMethod<prop extends keyof TalerMerchantHttpClient> = FailCasesByMethod<TalerMerchantHttpClient, prop>

/**
 */
export class TalerMerchantHttpClient {
  httpLib: HttpRequestLibrary;
  public readonly PROTOCOL_VERSION = "5:0:1";

  constructor(
    readonly baseUrl: string,
    httpClient?: HttpRequestLibrary,
  ) {
    this.httpLib = httpClient ?? createPlatformHttpLib();
  }

  isCompatible(version: string): boolean {
    const compare = LibtoolVersion.compare(this.PROTOCOL_VERSION, version)
    return compare?.compatible ?? false
  }
  /**
   * https://docs.taler.net/core/api-merchant.html#get--config
   * 
   */
  async getConfig() {
    const url = new URL(`config`, this.baseUrl);
    const resp = await this.httpLib.fetch(url.href, {
      method: "GET"
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForMerchantConfig())
      default: return opUnknownFailure(resp, await resp.text())
    }
  }

}