import { HttpRequestLibrary, makeBasicAuthHeader } from "../http-common.js";
import { HttpStatusCode } from "../http-status-codes.js";
import { createPlatformHttpLib } from "../http.js";
import { FailCasesByMethod, ResultByMethod, opFixedSuccess, opKnownFailure, opSuccess, opUnknownFailure } from "../operation.js";
import { PaginationParams, TalerWireGatewayApi, codecForAddIncomingResponse, codecForIncomingHistory, codecForOutgoingHistory, codecForTransferResponse } from "./types.js";
import { addPaginationParams } from "./utils.js";

export type TalerWireGatewayResultByMethod<prop extends keyof TalerWireGatewayHttpClient> = ResultByMethod<TalerWireGatewayHttpClient, prop>
export type TalerWireGatewayErrorsByMethod<prop extends keyof TalerWireGatewayHttpClient> = FailCasesByMethod<TalerWireGatewayHttpClient, prop>

/**
 * The API is used by the exchange to trigger transactions and query 
 * incoming transactions, as well as by the auditor to query incoming 
 * and outgoing transactions.
 * 
 * https://docs.taler.net/core/api-bank-wire.html
 */
export class TalerWireGatewayHttpClient {
  httpLib: HttpRequestLibrary;

  constructor(
    readonly baseUrl: string,
    readonly username: string,
    httpClient?: HttpRequestLibrary,
  ) {
    this.httpLib = httpClient ?? createPlatformHttpLib();
  }

  /**
   * https://docs.taler.net/core/api-bank-wire.html#post-$BASE_URL-transfer
   * 
   */
  async transfer(
    auth: string,
    body: TalerWireGatewayApi.TransferRequest,
  ) {
    const url = new URL(`transfer`, this.baseUrl);
    const resp = await this.httpLib.fetch(url.href, {
      method: "POST",
      headers: {
        Authorization: makeBasicAuthHeader(this.username, auth),
      },
      body
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForTransferResponse())
      case HttpStatusCode.BadRequest: return opKnownFailure("invalid-input", resp);
      case HttpStatusCode.Unauthorized: return opKnownFailure("unauthorized", resp);
      case HttpStatusCode.NotFound: return opKnownFailure("not-found", resp);
      case HttpStatusCode.Conflict: return opKnownFailure("request-uid-already-used", resp);
      default: return opUnknownFailure(resp, await resp.text())
    }
  }

  /**
   * https://docs.taler.net/core/api-bank-wire.html#get-$BASE_URL-history-incoming
   * 
   */
  async getHistoryIncoming(auth: string, pagination?: PaginationParams) {
    const url = new URL(`history/incoming`, this.baseUrl);
    addPaginationParams(url, pagination)
    const resp = await this.httpLib.fetch(url.href, {
      method: "GET",
      headers: {
        Authorization: makeBasicAuthHeader(this.username, auth),
      }
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForIncomingHistory())
      case HttpStatusCode.NoContent: return opFixedSuccess({ incoming_transactions: [], credit_account: "" })
      case HttpStatusCode.BadRequest: return opKnownFailure("invalid-input", resp);
      case HttpStatusCode.Unauthorized: return opKnownFailure("unauthorized", resp);
      case HttpStatusCode.NotFound: return opKnownFailure("not-found", resp);
      default: return opUnknownFailure(resp, await resp.text())
    }
    // return readSuccessResponseJsonOrThrow(resp, codecForIncomingHistory());
  }
  /**
   * https://docs.taler.net/core/api-bank-wire.html#get-$BASE_URL-history-outgoing
   * 
   */
  async getHistoryOutgoing(auth: string, pagination?: PaginationParams) {
    const url = new URL(`history/outgoing`, this.baseUrl);
    addPaginationParams(url, pagination)
    const resp = await this.httpLib.fetch(url.href, {
      method: "GET",
      headers: {
        Authorization: makeBasicAuthHeader(this.username, auth),
      }
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForOutgoingHistory())
      case HttpStatusCode.NoContent: return opFixedSuccess({ outgoing_transactions: [], debit_account: "" })
      case HttpStatusCode.BadRequest: return opKnownFailure("invalid-input", resp);
      case HttpStatusCode.Unauthorized: return opKnownFailure("unauthorized", resp);
      case HttpStatusCode.NotFound: return opKnownFailure("not-found", resp);
      default: return opUnknownFailure(resp, await resp.text())
    }
  }
  /**
   * https://docs.taler.net/core/api-bank-wire.html#post-$BASE_URL-admin-add-incoming
   * 
   */
  async addIncoming(auth: string, body: TalerWireGatewayApi.AddIncomingRequest,) {
    const url = new URL(`admin/add-incoming`, this.baseUrl);
    const resp = await this.httpLib.fetch(url.href, {
      method: "POST",
      headers: {
        Authorization: makeBasicAuthHeader(this.username, auth),
      },
      body
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForAddIncomingResponse())
      case HttpStatusCode.NoContent: return opFixedSuccess({ outgoing_transactions: [], debit_account: "" })
      case HttpStatusCode.BadRequest: return opKnownFailure("invalid-input", resp);
      case HttpStatusCode.Unauthorized: return opKnownFailure("unauthorized", resp);
      case HttpStatusCode.NotFound: return opKnownFailure("not-found", resp);
      case HttpStatusCode.Conflict: return opKnownFailure("reserve-id-already-used", resp);
      default: return opUnknownFailure(resp, await resp.text())
    }
  }
}

