import { HttpRequestLibrary, makeBasicAuthHeader, readSuccessResponseJsonOrThrow } from "../http-common.js";
import { HttpStatusCode } from "../http-status-codes.js";
import { createPlatformHttpLib } from "../http.js";
import { FailCasesByMethod, ResultByMethod, opKnownFailure, opSuccess, opUnknownFailure } from "../operation.js";
import { PaginationParams, TalerRevenueApi, codecForMerchantIncomingHistory } from "./types.js";
import { addPaginationParams } from "./utils.js";

export type TalerBankRevenueResultByMethod<prop extends keyof TalerRevenueHttpClient> = ResultByMethod<TalerRevenueHttpClient, prop>
export type TalerBankRevenueErrorsByMethod<prop extends keyof TalerRevenueHttpClient> = FailCasesByMethod<TalerRevenueHttpClient, prop>

/**
 * The API is used by the merchant (or other parties) to query 
 * for incoming transactions to their account.
 */
export class TalerRevenueHttpClient {
  httpLib: HttpRequestLibrary;

  constructor(
    readonly baseUrl: string,
    readonly username: string,
    httpClient?: HttpRequestLibrary,
  ) {
    this.httpLib = httpClient ?? createPlatformHttpLib();
  }

  /**
   * https://docs.taler.net/core/api-bank-revenue.html#get-$BASE_URL-history
   * 
   * @returns 
   */
  async getHistory(auth: string, pagination?: PaginationParams) {
    const url = new URL(`history`, this.baseUrl);
    addPaginationParams(url, pagination)
    const resp = await this.httpLib.fetch(url.href, {
      method: "GET",
      headers: {
        Authorization: makeBasicAuthHeader(this.username, auth),
      }
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForMerchantIncomingHistory())
      case HttpStatusCode.BadRequest: return opKnownFailure("invalid-input", resp);
      case HttpStatusCode.Unauthorized: return opKnownFailure("unauthorized", resp);
      case HttpStatusCode.NotFound: return opKnownFailure("endpoint-wrong-or-username-wrong", resp);
      default: return opUnknownFailure(resp, await resp.text())
    }
  }
}