import { HttpRequestLibrary, readSuccessResponseJsonOrThrow } from "../http-common.js";
import { HttpStatusCode } from "../http-status-codes.js";
import { createPlatformHttpLib } from "../http.js";
import { FailCasesByMethod, ResultByMethod, opKnownFailure, opSuccess, opUnknownFailure } from "../operation.js";
import { TalerErrorCode } from "../taler-error-codes.js";
import { codecForTalerErrorDetail } from "../wallet-types.js";
import {
  TalerBankIntegrationApi,
  codecForBankWithdrawalOperationPostResponse,
  codecForBankWithdrawalOperationStatus,
  codecForIntegrationBankConfig
} from "./types.js";

export type TalerBankIntegrationResultByMethod<prop extends keyof TalerBankIntegrationHttpClient> = ResultByMethod<TalerBankIntegrationHttpClient, prop>
export type TalerBankIntegrationErrorsByMethod<prop extends keyof TalerBankIntegrationHttpClient> = FailCasesByMethod<TalerBankIntegrationHttpClient, prop>

/**
 * The API is used by the wallets.
 */
export class TalerBankIntegrationHttpClient {
  httpLib: HttpRequestLibrary;

  constructor(
    readonly baseUrl: string,
    httpClient?: HttpRequestLibrary,
  ) {
    this.httpLib = httpClient ?? createPlatformHttpLib();
  }

  /**
   * https://docs.taler.net/core/api-bank-integration.html#get--config
   * 
   */
  async getConfig() {
    const url = new URL(`config`, this.baseUrl);
    const resp = await this.httpLib.fetch(url.href, {
      method: "GET"
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForIntegrationBankConfig())
      default: return opUnknownFailure(resp, await resp.text())
    }
  }

  /**
   * https://docs.taler.net/core/api-bank-integration.html#get-$BANK_API_BASE_URL-withdrawal-operation-$wopid
   * 
   */
  async getWithdrawalOperationById(woid: string, timeoutMs?: number) {
    const url = new URL(`withdrawal-operation/${woid}`, this.baseUrl);
    if (timeoutMs) {
      url.searchParams.set("long_poll_ms", String(timeoutMs))
    }
    const resp = await this.httpLib.fetch(url.href, {
      method: "GET"
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForBankWithdrawalOperationStatus())
      case HttpStatusCode.NotFound: return opKnownFailure("not-found", resp)
      default: return opUnknownFailure(resp, await resp.text())
    }
  }

  /**
   * https://docs.taler.net/core/api-bank-integration.html#post-$BANK_API_BASE_URL-withdrawal-operation-$wopid
   * 
   */
  async completeWithdrawalOperationById(woid: string, body: TalerBankIntegrationApi.BankWithdrawalOperationPostRequest) {
    const url = new URL(`withdrawal-operation/${woid}`, this.baseUrl);
    const resp = await this.httpLib.fetch(url.href, {
      method: "POST",
      body,
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForBankWithdrawalOperationPostResponse())
      case HttpStatusCode.NotFound: return opKnownFailure("not-found", resp)
      case HttpStatusCode.Conflict: {
        const body = await resp.json()
        const details = codecForTalerErrorDetail().decode(body)
        switch (details.code) {
          case TalerErrorCode.BANK_WITHDRAWAL_OPERATION_RESERVE_SELECTION_CONFLICT: return opKnownFailure("already-selected", resp);
          case TalerErrorCode.BANK_DUPLICATE_RESERVE_PUB_SUBJECT: return opKnownFailure("duplicated-reserve-id", resp);
          // case TalerErrorCode.BANK_ACCOUNT_NOT_FOUND: return opKnownFailure("account-not-found", resp);
          case TalerErrorCode.BANK_ACCOUNT_IS_NOT_EXCHANGE: return opKnownFailure("account-not-exchange", resp);
          default: return opUnknownFailure(resp, body)
        }
      }
      default: return opUnknownFailure(resp, await resp.text())
    }
  }

}

