/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { HttpStatusCode } from "../http-status-codes.js";
import { HttpRequestLibrary, createPlatformHttpLib, makeBasicAuthHeader } from "../http.js";
import { LibtoolVersion } from "../libtool-version.js";
import { opEmptySuccess, opKnownFailure, opSuccess, opUnknownFailure } from "../operation.js";
import { AccessToken, TalerAuthentication, codecForTokenSuccessResponse } from "./types.js";
import { makeBearerTokenAuthHeader } from "./utils.js";

export class TalerAuthenticationHttpClient {
  public readonly PROTOCOL_VERSION = "0:0:0";

  httpLib: HttpRequestLibrary;

  constructor(
    readonly baseUrl: string,
    readonly username: string,
    httpClient?: HttpRequestLibrary,
  ) {
    this.httpLib = httpClient ?? createPlatformHttpLib();
  }

  isCompatible(version: string): boolean {
    const compare = LibtoolVersion.compare(this.PROTOCOL_VERSION, version)
    return compare?.compatible ?? false
  }

  /**
   * https://docs.taler.net/core/api-corebank.html#post--accounts-$USERNAME-token
   * 
   * @returns 
   */
  async createAccessToken(
    password: string,
    body: TalerAuthentication.TokenRequest,
  ) {
    const url = new URL(`token`, this.baseUrl);
    const resp = await this.httpLib.fetch(url.href, {
      method: "POST",
      headers: {
        Authorization: makeBasicAuthHeader(this.username, password),
      },
      body
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opSuccess(resp, codecForTokenSuccessResponse())
      //FIXME: missing in docs
      case HttpStatusCode.Unauthorized: return opKnownFailure("wrong-credentials", resp)
      case HttpStatusCode.NotFound: return opKnownFailure("not-found", resp)
      default: return opUnknownFailure(resp, await resp.text())
    }
  }

  async deleteAccessToken(token: AccessToken) {
    const url = new URL(`token`, this.baseUrl);
    const resp = await this.httpLib.fetch(url.href, {
      method: "DELETE",
      headers: {
        Authorization: makeBearerTokenAuthHeader(token),
      }
    });
    switch (resp.status) {
      case HttpStatusCode.Ok: return opEmptySuccess()
      //FIXME: missing in docs
      case HttpStatusCode.NotFound: return opKnownFailure("not-found", resp)
      default: return opUnknownFailure(resp, await resp.text())
    }
  }

}