/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  TalerCorebankApiClient,
  Duration,
  NotificationType,
  TransactionMajorState,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig, defaultCoinConfig } from "../harness/denomStructures.js";
import {
  ExchangeService,
  FakebankService,
  GlobalTestState,
  MerchantService,
  WalletClient,
  WalletService,
  generateRandomTestIban,
  setupDb,
} from "../harness/harness.js";

/**
 * Test for wallet-core notifications.
 */
export async function runWalletNotificationsTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await FakebankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    currency: "TESTKUDOS",
    httpPort: 8083,
    database: db.connStr,
  });

  const exchangeBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );
  exchange.addBankAccount("1", exchangeBankAccount);

  bank.setSuggestedExchange(exchange, exchangeBankAccount.accountPaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const coinConfig: CoinConfig[] = defaultCoinConfig.map((x) => x("TESTKUDOS"));
  exchange.addCoinConfigList(coinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  // Fakebank uses x-taler-bank, but merchant is configured to only accept sepa!
  const label = "mymerchant";
  await merchant.addInstanceWithWireAccount({
    id: "default",
    name: "Default Instance",
    paytoUris: [
      `payto://iban/SANDBOXX/${generateRandomTestIban(label)}?receiver-name=${label}`,
    ],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  console.log("setup done!");

  const walletService = new WalletService(t, {
    name: "wallet",
    useInMemoryDb: true,
  });
  await walletService.start();
  await walletService.pingUntilAvailable();

  const walletClient = new WalletClient({
    unixPath: walletService.socketPath,
    onNotification(n) {
      console.log("got notification", n);
    },
  });
  await walletClient.connect();
  await walletClient.client.call(WalletApiOperation.InitWallet, {
    skipDefaults: true,
  });

  const bankAccessApiClient = new TalerCorebankApiClient(
    bank.corebankApiBaseUrl,
  );
  const user = await bankAccessApiClient.createRandomBankUser();
  bankAccessApiClient.setAuth(user);
  const wop = await bankAccessApiClient.createWithdrawalOperation(
    user.username,
    "TESTKUDOS:20",
  );

  // Hand it to the wallet

  await walletClient.client.call(
    WalletApiOperation.GetWithdrawalDetailsForUri,
    {
      talerWithdrawUri: wop.taler_withdraw_uri,
    },
  );

  // Withdraw (AKA select)

  const acceptRes = await walletClient.client.call(
    WalletApiOperation.AcceptBankIntegratedWithdrawal,
    {
      exchangeBaseUrl: exchange.baseUrl,
      talerWithdrawUri: wop.taler_withdraw_uri,
    },
  );

  const withdrawalFinishedReceivedPromise =
    walletClient.waitForNotificationCond((x) => {
      return (
        x.type === NotificationType.TransactionStateTransition &&
        x.newTxState.major === TransactionMajorState.Done &&
        x.transactionId === acceptRes.transactionId
      );
    });

  // Confirm it

  await bankAccessApiClient.confirmWithdrawalOperation(user.username, {
    withdrawalOperationId: wop.withdrawal_id,
  });

  await withdrawalFinishedReceivedPromise;
}

runWalletNotificationsTest.suites = ["wallet"];
