/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  TalerCorebankApiClient,
  MerchantApiClient,
  TransactionMajorState,
  WireGatewayApiClient,
  AmountString,
} from "@gnu-taler/taler-util";
import {
  WalletApiOperation,
} from "@gnu-taler/taler-wallet-core";
import { GlobalTestState, getWireMethodForTest } from "../harness/harness.js";
import { createSimpleTestkudosEnvironmentV2 } from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runTippingTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant, exchangeBankAccount } =
    await createSimpleTestkudosEnvironmentV2(t);

  const bankAccessApiClient = new TalerCorebankApiClient(
    bank.corebankApiBaseUrl,
  );
  const mbu = await bankAccessApiClient.createRandomBankUser();

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  const tipReserveResp = await merchantClient.createTippingReserve({
    exchange_url: exchange.baseUrl,
    initial_balance: "TESTKUDOS:10" as AmountString,
    wire_method: getWireMethodForTest(),
  });

  console.log("tipReserveResp:", tipReserveResp);

  t.assertDeepEqual(
    tipReserveResp.accounts[0].payto_uri,
    exchangeBankAccount.accountPaytoUri,
  );

  const wireGatewayApiClient = new WireGatewayApiClient(
    exchangeBankAccount.wireGatewayApiBaseUrl,
    {
      auth: {
        username: exchangeBankAccount.accountName,
        password: exchangeBankAccount.accountPassword,
      },
    },
  );

  await wireGatewayApiClient.adminAddIncoming({
    amount: "TESTKUDOS:10",
    debitAccountPayto: mbu.accountPaytoUri,
    reservePub: tipReserveResp.reserve_pub,
  });

  await exchange.runWirewatchOnce();

  await merchant.stop();
  await merchant.start();
  await merchant.pingUntilAvailable();

  const r = await merchantClient.queryTippingReserves();
  console.log("tipping reserves:", JSON.stringify(r, undefined, 2));

  t.assertTrue(r.reserves.length === 1);
  t.assertDeepEqual(
    r.reserves[0].exchange_initial_amount,
    r.reserves[0].merchant_initial_amount,
  );

  const tip = await merchantClient.giveTip({
    amount: "TESTKUDOS:5" as AmountString,
    justification: "why not?",
    next_url: "https://example.com/after-tip",
  });

  console.log("created tip", tip);

  const doTip = async (): Promise<void> => {
    const ptr = await walletClient.call(WalletApiOperation.PrepareReward, {
      talerRewardUri: tip.taler_reward_uri,
    });

    console.log(ptr);

    t.assertAmountEquals(ptr.rewardAmountRaw, "TESTKUDOS:5");
    t.assertAmountEquals(ptr.rewardAmountEffective, "TESTKUDOS:4.85");

    await walletClient.call(WalletApiOperation.AcceptReward, {
      walletRewardId: ptr.walletRewardId,
    });

    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );

    const bal = await walletClient.call(WalletApiOperation.GetBalances, {});

    console.log(bal);

    t.assertAmountEquals(bal.balances[0].available, "TESTKUDOS:4.85");

    const txns = await walletClient.call(
      WalletApiOperation.GetTransactions,
      {},
    );

    console.log("Transactions:", JSON.stringify(txns, undefined, 2));

    t.assertDeepEqual(txns.transactions[0].type, "reward");
    t.assertDeepEqual(
      txns.transactions[0].txState.major,
      TransactionMajorState.Done,
    );
    t.assertAmountEquals(
      txns.transactions[0].amountEffective,
      "TESTKUDOS:4.85",
    );
    t.assertAmountEquals(txns.transactions[0].amountRaw, "TESTKUDOS:5.0");
  };

  // Check twice so make sure tip handling is idempotent
  await doTip();
  await doTip();
}

runTippingTest.suites = ["wallet", "wallet-tipping"];
