/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  ConfirmPayResultType,
  Duration,
  durationFromSpec,
  MerchantApiClient,
  PreparePayResultType,
} from "@gnu-taler/taler-util";
import {
  PendingOperationsResponse,
  WalletApiOperation,
} from "@gnu-taler/taler-wallet-core";
import { makeNoFeeCoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  ExchangeService,
  generateRandomPayto,
  GlobalTestState,
  MerchantService,
  setupDb,
} from "../harness/harness.js";
import {
  applyTimeTravelV2,
  createWalletDaemonWithClient,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Basic time travel test.
 */
export async function runTimetravelAutorefreshTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    currency: "TESTKUDOS",
    httpPort: 8083,
    database: db.connStr,
  });

  const exchangeBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );
  exchange.addBankAccount("1", exchangeBankAccount);

  bank.setSuggestedExchange(exchange, exchangeBankAccount.accountPaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  exchange.addCoinConfigList(makeNoFeeCoinConfig("TESTKUDOS"));

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  await merchant.addInstanceWithWireAccount({
    id: "default",
    name: "Default Instance",
    paytoUris: [generateRandomPayto("merchant-default")],
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [generateRandomPayto("minst1")],
  });

  console.log("setup done!");

  const { walletClient } = await createWalletDaemonWithClient(t, {
    name: "w1",
  });

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:15",
  });
  await wres.withdrawalFinishedCond;

  // Travel into the future, the deposit expiration is two years
  // into the future.
  console.log("applying first time travel");
  await applyTimeTravelV2(
    Duration.toMilliseconds(durationFromSpec({ days: 400 })),
    {
      walletClient,
      exchange,
      merchant,
    },
  );

  let p: PendingOperationsResponse;
  p = await walletClient.call(WalletApiOperation.GetPendingOperations, {});

  console.log("pending operations after first time travel");
  console.log(JSON.stringify(p, undefined, 2));

  await walletClient.call(WalletApiOperation.TestingWaitTasksProcessed, {});
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const wres2 = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });
  await wres2.withdrawalFinishedCond;

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // Travel into the future, the deposit expiration is two years
  // into the future.
  console.log("applying second time travel");
  await applyTimeTravelV2(
    Duration.toMilliseconds(durationFromSpec({ years: 2, months: 6 })),
    {
      walletClient,
      exchange,
      merchant,
    },
  );

  await walletClient.call(WalletApiOperation.TestingWaitTasksProcessed, {});
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // At this point, the original coins should've been refreshed.
  // It would be too late to refresh them now, as we're past
  // the two year deposit expiration.

  const orderResp = await merchantClient.createOrder({
    order: {
      fulfillment_url: "http://example.com",
      summary: "foo",
      amount: "TESTKUDOS:30",
    },
  });

  const orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
    instance: "default",
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  const r = await walletClient.call(WalletApiOperation.PreparePayForUri, {
    talerPayUri: orderStatus.taler_pay_uri,
  });

  console.log(r);

  t.assertTrue(r.status === PreparePayResultType.PaymentPossible);

  const cpr = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: r.transactionId,
  });

  t.assertTrue(cpr.type === ConfirmPayResultType.Done);
}

runTimetravelAutorefreshTest.suites = ["wallet"];
