/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Amounts,
  Duration,
  MerchantApiClient,
  TransactionType,
  durationFromSpec,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState, delayMs } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runRefundIncrementalTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV2(t);

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  // Set up order.

  const orderResp = await merchantClient.createOrder({
    order: {
      summary: "Buy me!",
      amount: "TESTKUDOS:10",
      fulfillment_url: "taler://fulfillment-success/thx",
    },
    refund_delay: Duration.toTalerProtocolDuration(
      durationFromSpec({ minutes: 5 }),
    ),
  });

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  // Make wallet pay for the order

  const r1 = await walletClient.call(WalletApiOperation.PreparePayForUri, {
    talerPayUri: orderStatus.taler_pay_uri,
  });

  await walletClient.call(WalletApiOperation.ConfirmPay, {
    proposalId: r1.proposalId,
  });

  // Check if payment was successful.

  orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "paid");

  let ref = await merchantClient.giveRefund({
    amount: "TESTKUDOS:2.5",
    instance: "default",
    justification: "foo",
    orderId: orderResp.order_id,
  });

  console.log("first refund increase response", ref);

  {
    let wr = await walletClient.call(WalletApiOperation.StartRefundQuery, {
      transactionId: r1.transactionId,
    });
    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
    console.log(wr);
    const txs = await walletClient.call(WalletApiOperation.GetTransactions, {});
    console.log(
      "transactions after applying first refund:",
      JSON.stringify(txs, undefined, 2),
    );
  }

  // Wait at least a second, because otherwise the increased
  // refund will be grouped with the previous one.
  await delayMs(1200);

  ref = await merchantClient.giveRefund({
    amount: "TESTKUDOS:5",
    instance: "default",
    justification: "bar",
    orderId: orderResp.order_id,
  });

  console.log("second refund increase response", ref);

  // Wait at least a second, because otherwise the increased
  // refund will be grouped with the previous one.
  await delayMs(1200);

  ref = await merchantClient.giveRefund({
    amount: "TESTKUDOS:10",
    instance: "default",
    justification: "bar",
    orderId: orderResp.order_id,
  });

  console.log("third refund increase response", ref);

  {
    let wr = await walletClient.call(WalletApiOperation.StartRefundQuery, {
      transactionId: r1.transactionId,
    });
    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
    console.log(wr);
  }

  orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "paid");

  t.assertAmountEquals(orderStatus.refund_amount, "TESTKUDOS:10");

  console.log(JSON.stringify(orderStatus, undefined, 2));

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(JSON.stringify(bal, undefined, 2));

  {
    const txs = await walletClient.call(WalletApiOperation.GetTransactions, {});
    console.log(JSON.stringify(txs, undefined, 2));

    const txTypes = txs.transactions.map((x) => x.type);
    t.assertDeepEqual(txTypes, ["withdrawal", "payment", "refund", "refund"]);

    for (const tx of txs.transactions) {
      if (tx.type !== TransactionType.Refund) {
        continue;
      }
      t.assertAmountLeq(tx.amountEffective, tx.amountRaw);
    }

    const raw = Amounts.sum(
      txs.transactions
        .filter((x) => x.type === TransactionType.Refund)
        .map((x) => x.amountRaw),
    ).amount;

    t.assertAmountEquals("TESTKUDOS:10", raw);

    const effective = Amounts.sum(
      txs.transactions
        .filter((x) => x.type === TransactionType.Refund)
        .map((x) => x.amountEffective),
    ).amount;

    t.assertAmountEquals("TESTKUDOS:8.59", effective);
  }

  await t.shutdown();
}

runRefundIncrementalTest.suites = ["wallet"];
