/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Duration,
  NotificationType,
  TalerUriAction,
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
  WalletNotification,
  j2s,
  stringifyTalerUri,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  createWalletDaemonWithClient,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Run a test for basic peer-push payments.
 */
export async function runPeerToPeerPushTest(t: GlobalTestState) {
  const { bank, exchange } = await createSimpleTestkudosEnvironmentV2(t);

  let allW1Notifications: WalletNotification[] = [];
  let allW2Notifications: WalletNotification[] = [];

  const w1 = await createWalletDaemonWithClient(t, {
    name: "w1",
    handleNotification(wn) {
      allW1Notifications.push(wn);
    },
  });
  const w2 = await createWalletDaemonWithClient(t, {
    name: "w2",
    handleNotification(wn) {
      allW2Notifications.push(wn);
    },
  });

  // Withdraw digital cash into the wallet.

  const withdrawRes = await withdrawViaBankV2(t, {
    walletClient: w1.walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await withdrawRes.withdrawalFinishedCond;

  const purse_expiration = AbsoluteTime.toProtocolTimestamp(
    AbsoluteTime.addDuration(
      AbsoluteTime.now(),
      Duration.fromSpec({ days: 2 }),
    ),
  );

  {
    const resp = await w1.walletClient.call(
      WalletApiOperation.InitiatePeerPushDebit,
      {
        partialContractTerms: {
          summary: "Hello World 😁😇",
          amount: "TESTKUDOS:5" as AmountString,
          purse_expiration,
        },
      },
    );

    console.log(resp);
  }

  await w1.walletClient.call(WalletApiOperation.TestingWaitRefreshesFinal, {});

  const resp = await w1.walletClient.call(
    WalletApiOperation.InitiatePeerPushDebit,
    {
      partialContractTerms: {
        summary: "Hello World 🥺",
        amount: "TESTKUDOS:5" as AmountString,
        purse_expiration,
      },
    },
  );

  console.log(resp);

  const peerPushReadyCond = w1.walletClient.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.TransactionStateTransition &&
      x.newTxState.major === TransactionMajorState.Pending &&
      x.newTxState.minor === TransactionMinorState.Ready &&
      x.transactionId === resp.transactionId,
  );

  await peerPushReadyCond;

  const txDetails = await w1.walletClient.call(
    WalletApiOperation.GetTransactionById,
    {
      transactionId: resp.transactionId,
    },
  );
  t.assertDeepEqual(txDetails.type, TransactionType.PeerPushDebit);
  t.assertTrue(!!txDetails.talerUri);

  const checkResp = await w2.walletClient.call(
    WalletApiOperation.PreparePeerPushCredit,
    {
      talerUri: txDetails.talerUri,
    },
  );

  console.log(checkResp);

  const acceptResp = await w2.walletClient.call(
    WalletApiOperation.ConfirmPeerPushCredit,
    {
      peerPushCreditId: checkResp.peerPushCreditId,
    },
  );

  console.log(acceptResp);

  const txn1 = await w1.walletClient.call(
    WalletApiOperation.GetTransactions,
    {},
  );
  const txn2 = await w2.walletClient.call(
    WalletApiOperation.GetTransactions,
    {},
  );

  console.log(`txn1: ${j2s(txn1)}`);
  console.log(`txn2: ${j2s(txn2)}`);

  const ex1 = await t.assertThrowsTalerErrorAsync(async () => {
    await w1.walletClient.call(WalletApiOperation.InitiatePeerPushDebit, {
      partialContractTerms: {
        summary: "(this will fail)",
        amount: "TESTKUDOS:15" as AmountString,
        purse_expiration,
      },
    });
  });

  console.log("got expected exception detail", j2s(ex1.errorDetail));
}

runPeerToPeerPushTest.suites = ["wallet"];
