/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Duration,
  j2s,
  NotificationType,
  TransactionMajorState,
  TransactionMinorState,
  WalletNotification,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  createWalletDaemonWithClient,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runPeerToPeerPullTest(t: GlobalTestState) {
  // Set up test environment

  const { bank, exchange } = await createSimpleTestkudosEnvironmentV2(t);

  let allW1Notifications: WalletNotification[] = [];
  let allW2Notifications: WalletNotification[] = [];

  const w1 = await createWalletDaemonWithClient(t, {
    name: "w1",
    persistent: true,
    handleNotification(wn) {
      allW1Notifications.push(wn);
    },
  });
  const w2 = await createWalletDaemonWithClient(t, {
    name: "w2",
    persistent: true,
    handleNotification(wn) {
      allW2Notifications.push(wn);
    },
  });

  // Withdraw digital cash into the wallet.
  const wallet1 = w1.walletClient;
  const wallet2 = w2.walletClient;

  const withdrawRes = await withdrawViaBankV2(t, {
    walletClient: wallet2,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await withdrawRes.withdrawalFinishedCond;

  const purseExpiration = AbsoluteTime.toProtocolTimestamp(
    AbsoluteTime.addDuration(
      AbsoluteTime.now(),
      Duration.fromSpec({ days: 2 }),
    ),
  );

  const resp = await wallet1.client.call(
    WalletApiOperation.InitiatePeerPullCredit,
    {
      exchangeBaseUrl: exchange.baseUrl,
      partialContractTerms: {
        summary: "Hello World",
        amount: "TESTKUDOS:5" as AmountString,
        purse_expiration: purseExpiration,
      },
    },
  );

  const peerPullCreditReadyCond = wallet1.waitForNotificationCond(
    (x) => x.type === NotificationType.TransactionStateTransition &&
      x.transactionId === resp.transactionId &&
      x.newTxState.major === TransactionMajorState.Pending &&
      x.newTxState.minor === TransactionMinorState.Ready,
  );

  await peerPullCreditReadyCond;

  const checkResp = await wallet2.client.call(
    WalletApiOperation.PreparePeerPullDebit,
    {
      talerUri: resp.talerUri,
    },
  );

  console.log(`checkResp: ${j2s(checkResp)}`);

  const peerPullCreditDoneCond = wallet1.waitForNotificationCond(
    (x) => x.type === NotificationType.TransactionStateTransition &&
      x.transactionId === resp.transactionId &&
      x.newTxState.major === TransactionMajorState.Done,
  );

  const peerPullDebitDoneCond = wallet2.waitForNotificationCond(
    (x) => x.type === NotificationType.TransactionStateTransition &&
      x.transactionId === checkResp.transactionId &&
      x.newTxState.major === TransactionMajorState.Done,
  );

  await wallet2.client.call(WalletApiOperation.ConfirmPeerPullDebit, {
    peerPullDebitId: checkResp.peerPullDebitId,
  });

  await peerPullCreditDoneCond;
  await peerPullDebitDoneCond;

  const txn1 = await wallet1.client.call(
    WalletApiOperation.GetTransactions,
    {},
  );

  const txn2 = await wallet2.client.call(
    WalletApiOperation.GetTransactions,
    {},
  );

  console.log(`txn1: ${j2s(txn1)}`);
  console.log(`txn2: ${j2s(txn2)}`);

  console.log(`w1 notifications: ${j2s(allW1Notifications)}`);

  // Check that we don't have an excessive number of notifications.
  t.assertTrue(allW1Notifications.length <= 60);
}

runPeerToPeerPullTest.suites = ["wallet"];
