/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  MerchantApiClient,
  PreparePayResultType,
  TalerErrorCode,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState, WalletCli } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
} from "../harness/helpers.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runPaymentClaimTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange, merchant } =
    await createSimpleTestkudosEnvironmentV2(t);

  const merchantClient = new MerchantApiClient(merchant.makeInstanceBaseUrl());

  const walletTwo = new WalletCli(t, "two");

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  // Set up order.

  const orderResp = await merchantClient.createOrder({
    order: {
      summary: "Buy me!",
      amount: "TESTKUDOS:5",
      fulfillment_url: "taler://fulfillment-success/thx",
    },
  });

  let orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "unpaid");

  const talerPayUri = orderStatus.taler_pay_uri;

  // Make wallet pay for the order

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri,
    },
  );

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.PaymentPossible,
  );

  t.assertThrowsTalerErrorAsync(async () => {
    await walletTwo.client.call(WalletApiOperation.PreparePayForUri, {
      talerPayUri,
    });
  });

  await walletClient.call(WalletApiOperation.ConfirmPay, {
    proposalId: preparePayResult.proposalId,
  });

  // Check if payment was successful.

  orderStatus = await merchantClient.queryPrivateOrderStatus({
    orderId: orderResp.order_id,
  });

  t.assertTrue(orderStatus.order_status === "paid");

  walletTwo.deleteDatabase();

  const err = await t.assertThrowsTalerErrorAsync(async () => {
    await walletTwo.client.call(WalletApiOperation.PreparePayForUri, {
      talerPayUri,
    });
  });

  t.assertTrue(err.hasErrorCode(TalerErrorCode.WALLET_ORDER_ALREADY_CLAIMED));

  await t.shutdown();
}

runPaymentClaimTest.suites = ["wallet"];
