/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  MerchantService,
  generateRandomPayto,
  setupDb,
} from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV2,
  withdrawViaBankV2,
  makeTestPaymentV2,
  createWalletDaemonWithClient,
} from "../harness/helpers.js";
import { Duration, j2s } from "@gnu-taler/taler-util";
import { defaultCoinConfig } from "../harness/denomStructures.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runMultiExchangeTest(t: GlobalTestState) {
  // Set up test environment
  const dbDefault = await setupDb(t);

  const dbExchangeTwo = await setupDb(t, {
    nameSuffix: "exchange2",
  });

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: dbDefault.connStr,
    httpPort: 8082,
  });

  const exchangeOne = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: dbDefault.connStr,
  });

  const exchangeTwo = ExchangeService.create(t, {
    name: "testexchange-2",
    currency: "TESTKUDOS",
    httpPort: 8281,
    database: dbExchangeTwo.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    currency: "TESTKUDOS",
    httpPort: 8083,
    database: dbDefault.connStr,
  });

  const exchangeOneBankAccount = await bank.createExchangeAccount(
    "myexchange",
    "x",
  );
  await exchangeOne.addBankAccount("1", exchangeOneBankAccount);

  const exchangeTwoBankAccount = await bank.createExchangeAccount(
    "myexchange2",
    "x",
  );
  await exchangeTwo.addBankAccount("1", exchangeTwoBankAccount);

  bank.setSuggestedExchange(
    exchangeOne,
    exchangeOneBankAccount.accountPaytoUri,
  );

  await bank.start();

  await bank.pingUntilAvailable();

  // Set up the first exchange

  exchangeOne.addOfferedCoins(defaultCoinConfig);
  await exchangeOne.start();
  await exchangeOne.pingUntilAvailable();

  // Set up the second exchange

  exchangeTwo.addOfferedCoins(defaultCoinConfig);
  await exchangeTwo.start();
  await exchangeTwo.pingUntilAvailable();

  // Start and configure merchant

  merchant.addExchange(exchangeOne);
  merchant.addExchange(exchangeTwo);

  await merchant.start();
  await merchant.pingUntilAvailable();

  await merchant.addInstanceWithWireAccount({
    id: "default",
    name: "Default Instance",
    paytoUris: [generateRandomPayto("merchant-default")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [generateRandomPayto("minst1")],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  const { walletClient, walletService } = await createWalletDaemonWithClient(
    t,
    { name: "wallet" },
  );

  console.log("setup done!");

  // Withdraw digital cash into the wallet.

  await withdrawViaBankV2(t, {
    walletClient,
    bank,
    exchange: exchangeOne,
    amount: "TESTKUDOS:20",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:5",
    fulfillment_url: "taler://fulfillment-success/thx",
  };

  console.log("making test payment");

  await makeTestPaymentV2(t, { walletClient, merchant, order });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
}

runMultiExchangeTest.suites = ["wallet"];
