/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { ErrorType, HttpError } from "@gnu-taler/web-util/browser";
import { h, VNode } from "preact";
import { useEffect, useState } from "preact/hooks";
import { Loading } from "../../../../components/exception/loading.js";
import { MerchantBackend } from "../../../../declaration.js";
import { useInstanceDetails } from "../../../../hooks/instance.js";
import { useInstanceTransfers } from "../../../../hooks/transfer.js";
import { ListPage } from "./ListPage.js";
import { HttpStatusCode } from "@gnu-taler/taler-util";
import { useInstanceBankAccounts } from "../../../../hooks/bank.js";

interface Props {
  onUnauthorized: () => VNode;
  onLoadError: (error: HttpError<MerchantBackend.ErrorDetail>) => VNode;
  onNotFound: () => VNode;
  onCreate: () => void;
}
interface Form {
  verified?: "yes" | "no";
  payto_uri?: string;
}

export default function ListTransfer({
  onUnauthorized,
  onLoadError,
  onCreate,
  onNotFound,
}: Props): VNode {
  const setFilter = (s?: "yes" | "no") => setForm({ ...form, verified: s });

  const [position, setPosition] = useState<string | undefined>(undefined);

  const instance = useInstanceBankAccounts();
  const accounts = !instance.ok
    ? []
    : instance.data.accounts.map((a) => a.payto_uri);
  const [form, setForm] = useState<Form>({ payto_uri: "" });

  const shoulUseDefaultAccount = accounts.length === 1
  useEffect(() => {
    if (shoulUseDefaultAccount) {
      setForm({...form, payto_uri: accounts[0]})
    }
  }, [shoulUseDefaultAccount])

  const isVerifiedTransfers = form.verified === "yes";
  const isNonVerifiedTransfers = form.verified === "no";
  const isAllTransfers = form.verified === undefined;

  const result = useInstanceTransfers(
    {
      position,
      payto_uri: form.payto_uri === "" ? undefined : form.payto_uri,
      verified: form.verified,
    },
    (id) => setPosition(id),
  );

  if (result.loading) return <Loading />;
  if (!result.ok) {
    if (
      result.type === ErrorType.CLIENT &&
      result.status === HttpStatusCode.Unauthorized
    )
      return onUnauthorized();
    if (
      result.type === ErrorType.CLIENT &&
      result.status === HttpStatusCode.NotFound
    )
      return onNotFound();
    return onLoadError(result);
  }

  return (
    <ListPage
      accounts={accounts}
      transfers={result.data.transfers}
      onLoadMoreBefore={
        result.isReachingStart ? result.loadMorePrev : undefined
      }
      onLoadMoreAfter={result.isReachingEnd ? result.loadMore : undefined}
      onCreate={onCreate}
      onDelete={() => {
        null;
      }}
      // position={position} setPosition={setPosition}
      onShowAll={() => setFilter(undefined)}
      onShowUnverified={() => setFilter("no")}
      onShowVerified={() => setFilter("yes")}
      isAllTransfers={isAllTransfers}
      isVerifiedTransfers={isVerifiedTransfers}
      isNonVerifiedTransfers={isNonVerifiedTransfers}
      payTo={form.payto_uri}
      onChangePayTo={(p) => setForm((v) => ({ ...v, payto_uri: p }))}
    />
  );
}
