import { Amounts, TalerError } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { ErrorLoading } from "@gnu-taler/web-util/browser";
import { Loading } from "@gnu-taler/web-util/browser";
import { useBankCoreApiContext } from "../../context/config.js";
import { useBusinessAccounts } from "../../hooks/circuit.js";
import { RenderAmount } from "../PaytoWireTransferForm.js";
import { assertUnreachable } from "../WithdrawalOperationPage.js";

interface Props {
  onCreateAccount: () => void;

  onShowAccountDetails: (aid: string) => void;
  onRemoveAccount: (aid: string) => void;
  onUpdateAccountPassword: (aid: string) => void;
  onShowCashoutForAccount: (aid: string) => void;
}

export function AccountList({ onRemoveAccount, onShowAccountDetails, onUpdateAccountPassword, onShowCashoutForAccount, onCreateAccount }: Props): VNode {
  const result = useBusinessAccounts();
  const { i18n } = useTranslationContext();
  const { config } = useBankCoreApiContext()

  if (!result) {
    return <Loading />
  }
  if (result instanceof TalerError) {
    return <ErrorLoading error={result} />
  }
  if (result.data.type === "fail") {
    switch (result.data.case) {
      case "unauthorized": return <Fragment />
      default: assertUnreachable(result.data.case)
    }
  }

  const { accounts } = result.data.body;
  return <Fragment>
    <div class="px-4 sm:px-6 lg:px-8 mt-4">
      <div class="sm:flex sm:items-center">
        <div class="sm:flex-auto">
          <h1 class="text-base font-semibold leading-6 text-gray-900">
            <i18n.Translate>Accounts</i18n.Translate>
          </h1>
          <p class="mt-2 text-sm text-gray-700">
            <i18n.Translate>A list of all business account in the bank.</i18n.Translate>
          </p>
        </div>
        <div class="mt-4 sm:ml-16 sm:mt-0 sm:flex-none">
          <button type="button" class="block rounded-md bg-indigo-600 px-3 py-2 text-center text-sm font-semibold text-white shadow-sm hover:bg-indigo-500 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600"
            onClick={(e) => {
              e.preventDefault()
              onCreateAccount()
            }}>
            <i18n.Translate>Create account</i18n.Translate>
          </button>
        </div>
      </div>
      <div class="mt-8 flow-root">
        <div class="-mx-4 -my-2 overflow-x-auto sm:-mx-6 lg:-mx-8">
          <div class="inline-block min-w-full py-2 align-middle sm:px-6 lg:px-8">
            {!accounts.length ? (
              <div></div>
            ) : (
              <table class="min-w-full divide-y divide-gray-300">
                <thead>
                  <tr>
                    <th scope="col" class="py-3.5 pl-4 pr-3 text-left text-sm font-semibold text-gray-900 sm:pl-0">{i18n.str`Username`}</th>
                    <th scope="col" class="px-3 py-3.5 text-left text-sm font-semibold text-gray-900">{i18n.str`Name`}</th>
                    <th scope="col" class="px-3 py-3.5 text-left text-sm font-semibold text-gray-900">{i18n.str`Balance`}</th>
                    <th scope="col" class="relative py-3.5 pl-3 pr-4 sm:pr-0">
                      <span class="sr-only">{i18n.str`Actions`}</span>
                    </th>
                  </tr>
                </thead>
                <tbody class="divide-y divide-gray-200">
                  {accounts.map((item, idx) => {
                    const balance = !item.balance
                      ? undefined
                      : Amounts.parse(item.balance.amount);
                    const noBalance = Amounts.isZero(item.balance.amount)
                    const balanceIsDebit =
                      item.balance &&
                      item.balance.credit_debit_indicator == "debit";

                    return <tr key={idx}>
                      <td class="whitespace-nowrap py-4 pl-4 pr-3 text-sm font-medium text-gray-900 sm:pl-0">
                        <a href="#" class="text-indigo-600 hover:text-indigo-900"
                          onClick={(e) => {
                            e.preventDefault();
                            onShowAccountDetails(item.username)
                          }}
                        >
                          {item.username}
                        </a>


                      </td>
                      <td class="whitespace-nowrap px-3 py-4 text-sm text-gray-500">
                        {item.name}
                      </td>
                      <td data-negative={noBalance ? undefined : balanceIsDebit ? "true" : "false"} class="whitespace-nowrap px-3 py-4 text-sm text-gray-500 data-[negative=false]:text-green-600 data-[negative=true]:text-red-600 ">
                        {!balance ? (
                          i18n.str`unknown`
                        ) : (
                          <span class="amount">
                            <RenderAmount value={balance} negative={balanceIsDebit} spec={config.currency_specification} />
                          </span>
                        )}
                      </td>
                      <td class="relative whitespace-nowrap py-4 pl-3 pr-4 text-right text-sm font-medium sm:pr-0">
                        <a href="#" class="text-indigo-600 hover:text-indigo-900"
                          onClick={(e) => {
                            e.preventDefault();
                            onUpdateAccountPassword(item.username)
                          }}
                        >
                          change password
                        </a>
                        <br />
                        {noBalance ?
                          <a href="#" class="text-indigo-600 hover:text-indigo-900" onClick={(e) => {
                            e.preventDefault();
                            onRemoveAccount(item.username)
                          }}
                          >
                            remove
                          </a>
                          : undefined}
                      </td>
                    </tr>
                  })}

                </tbody>
              </table>
            )}
          </div>
        </div>
      </div>
    </div>
  </Fragment>

}