import { PaytoString, TalerCorebankApi, buildPayto, parsePaytoUri, stringifyPaytoUri } from "@gnu-taler/taler-util";
import { CopyButton, ShowInputErrorLabel, useTranslationContext } from "@gnu-taler/web-util/browser";
import { ComponentChildren, Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { PartialButDefined, RecursivePartial, WithIntermediate, undefinedIfEmpty, validateIBAN } from "../../utils.js";
import { doAutoFocus } from "../PaytoWireTransferForm.js";
import { assertUnreachable } from "../WithdrawalOperationPage.js";

const IBAN_REGEX = /^[A-Z][A-Z0-9]*$/;
const EMAIL_REGEX =
  /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
const REGEX_JUST_NUMBERS_REGEX = /^\+[0-9 ]*$/;

export type AccountFormData = TalerCorebankApi.AccountData & { username: string }

/**
 * Create valid account object to update or create
 * Take template as initial values for the form
 * Purpose indicate if all field al read only (show), part of them (update)
 * or none (create)
 * @param param0
 * @returns
 */
export function AccountForm({
  template,
  username,
  purpose,
  onChange,
  focus,
  noCashout,
  children,
}: {
  focus?: boolean,
  children: ComponentChildren,
  username?: string,
  noCashout?: boolean,
  template: TalerCorebankApi.AccountData | undefined;
  onChange: (a: AccountFormData | undefined) => void;
  purpose: "create" | "update" | "show";
}): VNode {
  const initial = initializeFromTemplate(username, template);
  const [form, setForm] = useState(initial);
  const [errors, setErrors] = useState<
    RecursivePartial<typeof initial> | undefined
  >(undefined);
  const { i18n } = useTranslationContext();

  function updateForm(newForm: typeof initial): void {
    const parsed = !newForm.cashout_payto_uri
      ? undefined
      : buildPayto("iban", newForm.cashout_payto_uri, undefined);;

    const errors = undefinedIfEmpty<RecursivePartial<typeof initial>>({
      cashout_payto_uri: (!newForm.cashout_payto_uri
        ? undefined
        : !parsed
          ? i18n.str`does not follow the pattern`
          : !parsed.isKnown || parsed.targetType !== "iban"
            ? i18n.str`only "IBAN" target are supported`
            : !IBAN_REGEX.test(parsed.iban)
              ? i18n.str`IBAN should have just uppercased letters and numbers`
              : validateIBAN(parsed.iban, i18n)) as PaytoString,
      contact_data: undefinedIfEmpty({
        email: !newForm.contact_data?.email
          ? undefined
          : !EMAIL_REGEX.test(newForm.contact_data.email)
            ? i18n.str`it should be an email`
            : undefined,
        phone: !newForm.contact_data?.phone
          ? undefined
          : !newForm.contact_data.phone.startsWith("+")
            ? i18n.str`should start with +`
            : !REGEX_JUST_NUMBERS_REGEX.test(newForm.contact_data.phone)
              ? i18n.str`phone number can't have other than numbers`
              : undefined,
      }),
      name: !newForm.name ? i18n.str`required` : undefined,
      username: !newForm.username ? i18n.str`required` : undefined,
    });
    setErrors(errors);
    setForm(newForm);
    if (errors) {
      onChange(undefined)
    } else {
      const cashout = !newForm.cashout_payto_uri? undefined :buildPayto("iban", newForm.cashout_payto_uri, undefined)
      const account: AccountFormData = {
        ...newForm as any,
        cashout_payto_uri: !cashout ? undefined : stringifyPaytoUri(cashout)
      }
      onChange(account);
    }
  }

  return (
    <form
      class="bg-white shadow-sm ring-1 ring-gray-900/5 sm:rounded-xl md:col-span-2"
      autoCapitalize="none"
      autoCorrect="off"
      onSubmit={e => {
        e.preventDefault()
      }}
    >
      <div class="px-4 py-6 sm:p-8">
        <div class="grid max-w-2xl grid-cols-1 gap-x-6 gap-y-8 sm:grid-cols-6">

          <div class="sm:col-span-5">
            <label
              class="block text-sm font-medium leading-6 text-gray-900"
              for="username"
            >
              {i18n.str`Username`}
              {purpose === "create" && <b style={{ color: "red" }}> *</b>}
            </label>
            <div class="mt-2">
              <input
                ref={focus && purpose === "create" ? doAutoFocus : undefined}
                type="text"
                class="block w-full disabled:bg-gray-100 rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 data-[error=true]:ring-red-500 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
                name="username"
                id="username"
                data-error={!!errors?.username && form.username !== undefined}
                disabled={purpose !== "create"}
                value={form.username ?? ""}
                onChange={(e) => {
                  form.username = e.currentTarget.value;
                  updateForm(structuredClone(form));
                }}
                // placeholder=""
                autocomplete="off"
              />
              <ShowInputErrorLabel
                message={errors?.username}
                isDirty={form.username !== undefined}
              />
            </div>
            <p class="mt-2 text-sm text-gray-500" >
              <i18n.Translate>account identification in the bank</i18n.Translate>
            </p>
          </div>

          <div class="sm:col-span-5">
            <label
              class="block text-sm font-medium leading-6 text-gray-900"
              for="name"
            >
              {i18n.str`Name`}
              {purpose === "create" && <b style={{ color: "red" }}> *</b>}
            </label>
            <div class="mt-2">
              <input
                type="text"
                class="block w-full disabled:bg-gray-100 rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 data-[error=true]:ring-red-500 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
                name="name"
                data-error={!!errors?.name && form.name !== undefined}
                id="name"
                disabled={purpose === "show"}
                value={form.name ?? ""}
                onChange={(e) => {
                  form.name = e.currentTarget.value;
                  updateForm(structuredClone(form));
                }}
                // placeholder=""
                autocomplete="off"
              />
              <ShowInputErrorLabel
                message={errors?.name}
                isDirty={form.name !== undefined}
              />
            </div>
            <p class="mt-2 text-sm text-gray-500" >
              <i18n.Translate>name of the person owner the account</i18n.Translate>
            </p>
          </div>


          {purpose !== "create" && (<RenderPaytoDisabledField paytoURI={form.payto_uri as PaytoString} />)}

          <div class="sm:col-span-5">
            <label
              class="block text-sm font-medium leading-6 text-gray-900"
              for="email"
            >
              {i18n.str`Email`}
              {purpose === "create" && <b style={{ color: "red" }}> *</b>}
            </label>
            <div class="mt-2">
              <input
                type="email"
                class="block w-full disabled:bg-gray-100 rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 data-[error=true]:ring-red-500 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
                name="email"
                id="email"
                data-error={!!errors?.contact_data?.email && form.contact_data?.email !== undefined}
                disabled={purpose === "show"}
                value={form.contact_data?.email ?? ""}
                onChange={(e) => {
                  if (form.contact_data) {
                    form.contact_data.email = e.currentTarget.value;
                    if (!form.contact_data.email) {
                      form.contact_data.email = undefined
                    }
                    updateForm(structuredClone(form));
                  }
                }}
                autocomplete="off"
              />
              <ShowInputErrorLabel
                message={errors?.contact_data?.email}
                isDirty={form.contact_data?.email !== undefined}
              />
            </div>
          </div>

          <div class="sm:col-span-5">
            <label
              class="block text-sm font-medium leading-6 text-gray-900"
              for="phone"
            >
              {i18n.str`Phone`}
              {purpose === "create" && <b style={{ color: "red" }}> *</b>}
            </label>
            <div class="mt-2">
              <input
                type="text"
                class="block w-full disabled:bg-gray-100 rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 data-[error=true]:ring-red-500 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
                name="phone"
                id="phone"
                disabled={purpose === "show"}
                value={form.contact_data?.phone ?? ""}
                data-error={!!errors?.contact_data?.phone && form.contact_data?.phone !== undefined}
                onChange={(e) => {
                  if (form.contact_data) {
                    form.contact_data.phone = e.currentTarget.value;
                    if (!form.contact_data.email) {
                      form.contact_data.email = undefined
                    }
                    updateForm(structuredClone(form));
                  }
                }}
                // placeholder=""
                autocomplete="off"
              />
              <ShowInputErrorLabel
                message={errors?.contact_data?.phone}
                isDirty={form.contact_data?.phone !== undefined}
              />
            </div>
          </div>


          {!noCashout &&
            <div class="sm:col-span-5">
              <label
                class="block text-sm font-medium leading-6 text-gray-900"
                for="cashout"
              >
                {i18n.str`Cashout IBAN`}
                {purpose !== "show" && <b style={{ color: "red" }}> *</b>}
              </label>
              <div class="mt-2">
                <input
                  type="text"
                  ref={focus && purpose === "update" ? doAutoFocus : undefined}
                  data-error={!!errors?.cashout_payto_uri && form.cashout_payto_uri !== undefined}
                  class="block w-full disabled:bg-gray-100 rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 data-[error=true]:ring-red-500 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
                  name="cashout"
                  id="cashout"
                  disabled={purpose === "show"}
                  value={form.cashout_payto_uri ?? ""}
                  onChange={(e) => {
                    form.cashout_payto_uri = e.currentTarget.value as PaytoString;
                    if (!form.cashout_payto_uri) {
                      form.cashout_payto_uri= undefined
                    }
                    updateForm(structuredClone(form));
                  }}
                  autocomplete="off"
                />
                <ShowInputErrorLabel
                  message={errors?.cashout_payto_uri}
                  isDirty={form.cashout_payto_uri !== undefined}
                />
              </div>
              <p class="mt-2 text-sm text-gray-500" >
                <i18n.Translate>account number where the money is going to be sent when doing cashouts</i18n.Translate>
              </p>
            </div>
          }

        </div>
      </div>
      {children}
    </form>
  );
}

function initializeFromTemplate(
  username: string | undefined,
  account: TalerCorebankApi.AccountData | undefined,
): WithIntermediate<AccountFormData> {
  const emptyAccount = {
    cashout_payto_uri: undefined,
    contact_data: undefined,
    payto_uri: undefined,
    balance: undefined,
    debit_threshold: undefined,
    name: undefined,
  };
  const emptyContact = {
    email: undefined,
    phone: undefined,
  };

  const initial: PartialButDefined<TalerCorebankApi.AccountData> =
    structuredClone(account) ?? emptyAccount;
  if (typeof initial.contact_data === "undefined") {
    initial.contact_data = emptyContact;
  }
  if (initial.cashout_payto_uri) {
    const ac = parsePaytoUri(initial.cashout_payto_uri)
    if (ac?.isKnown && ac.targetType === "iban") {
      // we are using the cashout field for the iban number
      initial.cashout_payto_uri = ac.targetPath as any
    }
  }
  const result: WithIntermediate<AccountFormData> = initial as any // FIXME: check types
  result.username = username

  return initial as any;
}


function RenderPaytoDisabledField({ paytoURI }: { paytoURI: PaytoString | undefined }): VNode {
  const { i18n } = useTranslationContext()
  const payto = parsePaytoUri(paytoURI ?? "");
  if (payto?.isKnown) {
    if (payto.targetType === "iban") {
      const value = payto.iban;
      return <div class="sm:col-span-5">
        <label
          class="block text-sm font-medium leading-6 text-gray-900"
          for="internal-iban"
        >
          {i18n.str`Internal IBAN`}
        </label>
        <div class="mt-2">
          <div class="flex justify-between">
            <input
              type="text"
              class="mr-4 w-full block-inline  disabled:bg-gray-100 rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 data-[error=true]:ring-red-500 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
              name="internal-iban"
              id="internal-iban"
              disabled={true}
              value={value ?? ""}
            />
            <CopyButton
              class="p-2 rounded-full  text-black shadow-sm  focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 "
              getContent={() => value ?? ""}
            />
          </div>
        </div>
        <p class="mt-2 text-sm text-gray-500" >
          <i18n.Translate>international bank account number</i18n.Translate>
        </p>
      </div>
    }
    if (payto.targetType === "x-taler-bank") {
      const value = payto.account;
      return <div class="sm:col-span-5">
        <label
          class="block text-sm font-medium leading-6 text-gray-900"
          for="account-id"
        >
          {i18n.str`Account ID`}
        </label>
        <div class="mt-2">
          <div class="flex justify-between">
            <input
              type="text"
              class="mr-4 w-full block-inline  disabled:bg-gray-100 rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 data-[error=true]:ring-red-500 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
              name="account-id"
              id="account-id"
              disabled={true}
              value={value ?? ""}
            />
            <CopyButton
              class="p-2 rounded-full  text-black shadow-sm  focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 "
              getContent={() => value ?? ""}
            />
          </div>
        </div>
        <p class="mt-2 text-sm text-gray-500" >
          <i18n.Translate>internal account id</i18n.Translate>
        </p>
      </div>
    }
    if (payto.targetType === "bitcoin") {
      const value = payto.targetPath;
      return <div class="sm:col-span-5">
        <label
          class="block text-sm font-medium leading-6 text-gray-900"
          for="account-id"
        >
          {i18n.str`Bitcoin address`}
        </label>
        <div class="mt-2">
          <div class="flex justify-between">
            <input
              type="text"
              class="mr-4 w-full block-inline  disabled:bg-gray-100 rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 data-[error=true]:ring-red-500 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
              name="account-id"
              id="account-id"
              disabled={true}
              value={value ?? ""}
            />
            <CopyButton
              class="p-2 rounded-full  text-black shadow-sm  focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 "
              getContent={() => value ?? ""}
            />
          </div>
        </div>
        <p class="mt-2 text-sm text-gray-500" >
          <i18n.Translate>bitcoin address</i18n.Translate>
        </p>
      </div>
    }
    assertUnreachable(payto)
  }

  const value = paytoURI ?? ""
  return <div class="sm:col-span-5">
    <label
      class="block text-sm font-medium leading-6 text-gray-900"
      for="internal-payto"
    >
      {i18n.str`Internal account`}
    </label>
    <div class="mt-2">
      <div class="flex justify-between">
        <input
          type="text"
          class="mr-4 w-full block-inline  disabled:bg-gray-100 rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 data-[error=true]:ring-red-500 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
          name="internal-payto"
          id="internal-payto"
          disabled={true}
          value={value ?? ""}
        />
        <CopyButton
          class="p-2 rounded-full  text-black shadow-sm  focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 "
          getContent={() => value ?? ""}
        />
      </div>
    </div>
    <p class="mt-2 text-sm text-gray-500" >
      <i18n.Translate>generic payto URI</i18n.Translate>
    </p>
  </div>
}
