import { Amounts, TalerError } from "@gnu-taler/taler-util";
import { notifyInfo, useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { ErrorLoading } from "@gnu-taler/web-util/browser";
import { Loading } from "@gnu-taler/web-util/browser";
import { useAccountDetails } from "../hooks/access.js";
import { assertUnreachable } from "./WithdrawalOperationPage.js";
import { LoginForm } from "./LoginForm.js";
import { PaytoWireTransferForm } from "./PaytoWireTransferForm.js";
import { useBackendState } from "../hooks/backend.js";

export function WireTransfer({ toAccount, onRegister, onCancel, onSuccess }: { onSuccess?: () => void; toAccount?: string, onCancel?: () => void, onRegister?: () => void }): VNode {
  const { i18n } = useTranslationContext();
  const r = useBackendState();
  const account = r.state.status !== "loggedOut" ? r.state.username : "admin";
  const result = useAccountDetails(account);

  if (!result) {
    return <Loading />
  }
  if (result instanceof TalerError) {
    return <ErrorLoading error={result} />
  }
  if (result.type === "fail") {
    switch (result.case) {
      case "unauthorized": return <LoginForm reason="forbidden" />
      case "not-found": return <LoginForm reason="not-found" />
      default: assertUnreachable(result)
    }
  }
  const { body: data } = result;

  const balance = Amounts.parseOrThrow(data.balance.amount);
  const balanceIsDebit = data.balance.credit_debit_indicator == "debit";

  const debitThreshold = Amounts.parseOrThrow(data.debit_threshold);
  const limit = balanceIsDebit
    ? Amounts.sub(debitThreshold, balance).amount
    : Amounts.add(balance, debitThreshold).amount;
  if (!balance) return <Fragment />;
  return (
    <PaytoWireTransferForm
      title={i18n.str`Make a wire transfer`}
      toAccount={toAccount}
      limit={limit}
      onSuccess={() => {
        notifyInfo(i18n.str`Wire transfer created!`);
        if (onSuccess) onSuccess()
      }}
      onCancel={onCancel}
    />
  );
}
