/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { Logger, TalerError } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { Loading } from "@gnu-taler/web-util/browser";
import { Transactions } from "../components/Transactions/index.js";
import { usePublicAccounts } from "../hooks/access.js";

const logger = new Logger("PublicHistoriesPage");

interface Props { }

/** 
 * Show histories of public accounts.
 */
export function PublicHistoriesPage({ }: Props): VNode {
  const { i18n } = useTranslationContext();

  const result = usePublicAccounts();
  const firstAccount = result && !(result instanceof TalerError) && result.data.public_accounts.length > 0
    ? result.data.public_accounts[0].account_name
    : undefined;

  const [showAccount, setShowAccount] = useState(firstAccount);

  if (!result) {
    return <Loading />
  }
  if (result instanceof TalerError) {
    return <Loading />
  }

  const { data } = result;

  const txs: Record<string, h.JSX.Element> = {};
  const accountsBar = [];

  // Ask story of all the public accounts.
  for (const account of data.public_accounts) {
    logger.trace("Asking transactions for", account.account_name);
    const isSelected = account.account_name == showAccount;
    accountsBar.push(
      <li
        class={
          isSelected
            ? "pure-menu-selected pure-menu-item"
            : "pure-menu-item pure-menu"
        }
      >
        <a
          href="#"
          class="pure-menu-link"
          onClick={() => setShowAccount(account.account_name)}
        >
          {account.account_name}
        </a>
      </li>,
    );
    txs[account.account_name] = <Transactions account={account.account_name} />;
  }

  return (
    <Fragment>
      <h1 class="nav">{i18n.str`History of public accounts`}</h1>
      <section id="main">
        <article>
          <div class="pure-menu pure-menu-horizontal" name="accountMenu">
            <ul class="pure-menu-list">{accountsBar}</ul>
            {typeof showAccount !== "undefined" ? (
              txs[showAccount]
            ) : (
              <p>No public transactions found.</p>
            )}
            <br />
            <a href="/account" class="pure-button">
              Go back
            </a>
          </div>
        </article>
      </section>
    </Fragment>
  );
}
