/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { Attention } from "@gnu-taler/web-util/browser";
import { Transactions } from "../../components/Transactions/index.js";
import { usePreferences } from "../../hooks/preferences.js";
import { PaymentOptions } from "../PaymentOptions.js";
import { State } from "./index.js";
import { useCashouts, useOnePendingCashouts } from "../../hooks/circuit.js";
import { TalerError } from "@gnu-taler/taler-util";

export function InvalidIbanView({ error }: State.InvalidIban) {
  return (
    <div>Payto from server is not valid &quot;{error.payto_uri}&quot;</div>
  );
}

const IS_PUBLIC_ACCOUNT_ENABLED = false

function ShowDemoInfo(): VNode {
  const { i18n } = useTranslationContext();
  const [settings, updateSettings] = usePreferences();
  if (!settings.showDemoDescription) return <Fragment />
  return <Attention title={i18n.str`This is a demo bank`} onClose={() => {
    updateSettings("showDemoDescription", false);
  }}>
    {IS_PUBLIC_ACCOUNT_ENABLED ? (
      <i18n.Translate>
        This part of the demo shows how a bank that supports Taler
        directly would work. In addition to using your own bank
        account, you can also see the transaction history of some{" "}
        <a href="/public-accounts">Public Accounts</a>.
      </i18n.Translate>
    ) : (
      <i18n.Translate>
        This part of the demo shows how a bank that supports Taler
        directly would work.
      </i18n.Translate>
    )}
  </Attention>
}

export function ReadyView({ account, limit, goToConfirmOperation }: State.Ready): VNode<{}> {

  return <Fragment>
    <ShowDemoInfo />
    <PendingCashouts account={account}/>
    <PaymentOptions limit={limit} goToConfirmOperation={goToConfirmOperation} />
    <Transactions account={account} />
  </Fragment>;
}


function PendingCashouts({account}: {account: string}):VNode {
  const { i18n } = useTranslationContext();
  const result = useOnePendingCashouts(account)
  if (!result || result instanceof TalerError || result.type !== "ok" || !result.body) {
    return <Fragment />
  }

  return <Attention title={i18n.str`You have pending cashout operation to complete`} >
    <i18n.Translate>
      Cashout with subject "{result.body.subject}", look for the code and complete the operation <a target="_blank" rel="noreferrer noopener" class="font-semibold text-blue-700 hover:text-blue-600" href={`#/cashout/${result.body.id}`}>here</a>.
    </i18n.Translate>
  </Attention>
}