/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { Amounts, TalerError, parsePaytoUri } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { useAccountDetails } from "../../hooks/access.js";
import { assertUnreachable } from "../WithdrawalOperationPage.js";
import { Props, State } from "./index.js";

export function useComponentState({ account, goToConfirmOperation }: Props): State {
  const result = useAccountDetails(account);
  const { i18n } = useTranslationContext();

  if (!result) {
    return {
      status: "loading",
      error: undefined,
    };
  }

  if (result instanceof TalerError) {
    return {
      status: "loading-error",
      error: result,
    };
  }

  if (result.type === "fail") {
    switch (result.case) {
      case "unauthorized": return {
        status: "login",
        reason: "forbidden"
      }
      case "not-found": return {
        status: "login",
        reason: "not-found",
      }
      default: {
        assertUnreachable(result)
      }
    }
  }

  const { body: data } = result;

  const balance = Amounts.parseOrThrow(data.balance.amount);

  const debitThreshold = Amounts.parseOrThrow(data.debit_threshold);
  const payto = parsePaytoUri(data.payto_uri);

  if (!payto || !payto.isKnown || (payto.targetType !== "iban" && payto.targetType !== "x-taler-bank")) {
    return {
      status: "invalid-iban",
      error: data
    };
  }

  const balanceIsDebit = data.balance.credit_debit_indicator == "debit";
  const limit = balanceIsDebit
    ? Amounts.sub(debitThreshold, balance).amount
    : Amounts.add(balance, debitThreshold).amount;


  return {
    status: "ready",
    goToConfirmOperation,
    error: undefined,
    account,
    limit,
  };
}
