/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { StateUpdater } from "preact/hooks";
import { buildStorageKey, useLocalStorage } from "@gnu-taler/web-util/browser";
import { codecForBoolean } from "@gnu-taler/taler-util";
export type ValueOrFunction<T> = T | ((p: T) => T);

const calculateRootPath = () => {
  const rootPath =
    typeof window !== undefined
      ? window.location.origin + window.location.pathname
      : "/";
  return rootPath;
};

const BACKEND_URL_KEY = buildStorageKey("backend-url");
const TRIED_LOGIN_KEY = buildStorageKey("tried-login", codecForBoolean());

export function useBackendURL(
  url?: string,
): [string, boolean, StateUpdater<string>, () => void] {
  const { value, update: setter } = useLocalStorage(
    BACKEND_URL_KEY,
    url || calculateRootPath(),
  );

  const {
    value: triedToLog,
    update: setTriedToLog,
    reset: resetBackend,
  } = useLocalStorage(TRIED_LOGIN_KEY);

  const checkedSetter = (v: ValueOrFunction<string>) => {
    setTriedToLog(true);
    const computedValue =
      v instanceof Function ? v(value) : v.replace(/\/$/, "");
    return setter(computedValue);
  };

  return [value, !!triedToLog, checkedSetter, resetBackend];
}
