/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { TalerCorebankApi, TalerCoreBankHttpClient, TalerError } from "@gnu-taler/taler-util";
import { BrowserHttpLib, useTranslationContext } from "@gnu-taler/web-util/browser";
import { ComponentChildren, createContext, FunctionComponent, h, VNode } from "preact";
import { useContext, useEffect, useState } from "preact/hooks";
import { ErrorLoading } from "@gnu-taler/web-util/browser";

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

export type Type = {
  url: URL,
  config: TalerCorebankApi.Config,
  api: TalerCoreBankHttpClient,
};

const Context = createContext<Type>(undefined as any);

export const useBankCoreApiContext = (): Type => useContext(Context);

export type ConfigResult = undefined
  | { type: "ok", config: TalerCorebankApi.Config }
  | { type: "incompatible", result: TalerCorebankApi.Config, supported: string }
  | { type: "error", error: TalerError }

export const BankCoreApiProvider = ({
  baseUrl,
  children,
  frameOnError,
}: {
  baseUrl: string,
  children: ComponentChildren;
  frameOnError: FunctionComponent<{ children: ComponentChildren }>,
}): VNode => {
  const [checked, setChecked] = useState<ConfigResult>()
  const { i18n } = useTranslationContext();
  const url = new URL(baseUrl)
  const api = new TalerCoreBankHttpClient(url.href, new BrowserHttpLib())
  useEffect(() => {
    api.getConfig()
      .then((resp) => {
        if (api.isCompatible(resp.body.version)) {
          setChecked({ type: "ok", config: resp.body });
        } else {
          setChecked({ type: "incompatible", result: resp.body, supported: api.PROTOCOL_VERSION })
        }
      })
      .catch((error: unknown) => {
        if (error instanceof TalerError) {
          setChecked({ type: "error", error });
        }
      });
  }, []);

  if (checked === undefined) {
    return h(frameOnError, { children: h("div", {}, "loading...") })
  }
  if (checked.type === "error") {
    return h(frameOnError, { children: h(ErrorLoading, { error: checked.error, showDetail: true }) })
  }
  if (checked.type === "incompatible") {
    return h(frameOnError, { children: h("div", {}, i18n.str`the bank backend is not supported. supported version "${checked.supported}", server version "${checked.result.version}"`) })
  }
  const value: Type = {
    url, config: checked.config, api
  }
  return h(Context.Provider, {
    value,
    children,
  });
};

export const BankCoreApiProviderTesting = ({
  children,
  state,
  url,
}: {
  children: ComponentChildren;
  state: TalerCorebankApi.Config;
  url: string,
}): VNode => {
  const value: Type = {
    url: new URL(url),
    config: state,
    api: undefined as any,
  };

  return h(Context.Provider, {
    value,
    children,
  });
};
