/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  canonicalizeBaseUrl,
  getGlobalLogLevel,
  setGlobalLogLevelFromString
} from "@gnu-taler/taler-util";
import { Loading, TranslationProvider } from "@gnu-taler/web-util/browser";
import { Fragment, FunctionalComponent, h } from "preact";
import { SWRConfig } from "swr";
import { BackendStateProvider } from "../context/backend.js";
import { BankCoreApiProvider } from "../context/config.js";
import { strings } from "../i18n/strings.js";
import { BankUiSettings, fetchSettings } from "../settings.js";
import { Routing } from "../Routing.js";
import { BankFrame } from "../pages/BankFrame.js";
import { useEffect, useState } from "preact/hooks";
import { SettingsProvider } from "../context/settings.js";
const WITH_LOCAL_STORAGE_CACHE = false;

const App: FunctionalComponent = () => {
  const [settings, setSettings] = useState<BankUiSettings>()
  useEffect(() => {
    fetchSettings(setSettings)
  }, [])
  if (!settings) return <Loading />;
  
  const baseUrl = getInitialBackendBaseURL(settings.backendBaseURL);
  return (
    <SettingsProvider value={settings}>
      <TranslationProvider source={strings}>
        <BackendStateProvider>
          <BankCoreApiProvider baseUrl={baseUrl} frameOnError={BankFrame}>
            <SWRConfig
              value={{
                provider: WITH_LOCAL_STORAGE_CACHE
                  ? localStorageProvider
                  : undefined,
              }}
            >
              <Routing />
            </SWRConfig>
          </BankCoreApiProvider>
        </BackendStateProvider>
      </TranslationProvider >
    </SettingsProvider>
  );
};

(window as any).setGlobalLogLevelFromString = setGlobalLogLevelFromString;
(window as any).getGlobalLevel = getGlobalLogLevel;

function localStorageProvider(): Map<unknown, unknown> {
  const map = new Map(JSON.parse(localStorage.getItem("app-cache") || "[]"));

  window.addEventListener("beforeunload", () => {
    const appCache = JSON.stringify(Array.from(map.entries()));
    localStorage.setItem("app-cache", appCache);
  });
  return map;
}

export default App;

function getInitialBackendBaseURL(backendFromSettings: string | undefined): string {
  const overrideUrl =
    typeof localStorage !== "undefined"
      ? localStorage.getItem("bank-base-url")
      : undefined;
  let result: string;

  if (!overrideUrl) {
    //normal path
    if (!backendFromSettings) {
      console.error(
        "ERROR: backendBaseURL was overridden by a setting file and missing. Setting value to 'window.origin'",
      );
      result = window.origin
    } else {
      result = backendFromSettings;
    }
  } else {
    // testing/development path
    result = overrideUrl
  }
  try {
    return canonicalizeBaseUrl(result)
  } catch (e) {
    //fall back
    return canonicalizeBaseUrl(window.origin)
  }
}