/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { Amounts, TalerError, assertUnreachable } from "@gnu-taler/taler-util";
import { Attention, ErrorLoading, Loading, useTranslationContext } from "@gnu-taler/web-util/browser";
import { format } from "date-fns";
import { Fragment, h, VNode } from "preact";
import { RenderAmount } from "../../pages/PaytoWireTransferForm.js";
import { State } from "./index.js";
import { useConversionInfo } from "../../hooks/circuit.js";

export function LoadingUriView({ error }: State.LoadingUriError): VNode {
  const { i18n } = useTranslationContext();

  return (
    <div>
      <i18n.Translate>Could not load</i18n.Translate>
    </div>
  );
}
export function FailedView({ error }: State.Failed) {
  const { i18n } = useTranslationContext();
  switch (error.case) {
    case "cashout-not-supported": return <Attention type="danger"
      title={i18n.str`Cashout not supported.`}>
      <div class="mt-2 text-sm text-red-700">
        {error.detail.hint}
      </div>
    </Attention>
    case "account-not-found": return <Attention type="danger"
      title={i18n.str`Account not found.`}>
      <div class="mt-2 text-sm text-red-700">
        {error.detail.hint}
      </div>
    </Attention>
    default: assertUnreachable(error)
  }
}

export function ReadyView({ cashouts, onSelected }: State.Ready): VNode {
  const { i18n } = useTranslationContext();
  const resp = useConversionInfo();
  if (!resp) {
    return <Loading />
  }
  if (resp instanceof TalerError) {
    return <ErrorLoading error={resp} />
  }
  if (!cashouts.length) return <div />
  const txByDate = cashouts.reduce((prev, cur) => {
    const d = cur.creation_time.t_s === "never"
      ? ""
      : format(cur.creation_time.t_s * 1000, "dd/MM/yyyy")
    if (!prev[d]) {
      prev[d] = []
    }
    prev[d].push(cur)
    return prev
  }, {} as Record<string, typeof cashouts>)
  return (
    <div class="px-4 mt-4">
      <div class="sm:flex sm:items-center">
        <div class="sm:flex-auto">
          <h1 class="text-base font-semibold leading-6 text-gray-900"><i18n.Translate>Latest cashouts</i18n.Translate></h1>
        </div>
      </div>
      <div class="-mx-4 mt-5 ring-1 ring-gray-300 sm:mx-0 rounded-lg min-w-fit bg-white">
        <table class="min-w-full divide-y divide-gray-300">
          <thead>
            <tr>
              <th scope="col" class="                     pl-2 py-3.5 text-left text-sm font-semibold text-gray-900">{i18n.str`Created`}</th>
              <th scope="col" class="hidden sm:table-cell pl-2 py-3.5 text-left text-sm font-semibold text-gray-900">{i18n.str`Confirmed`}</th>
              <th scope="col" class="hidden sm:table-cell pl-2 py-3.5 text-left text-sm font-semibold text-gray-900">{i18n.str`Total debit`}</th>
              <th scope="col" class="hidden sm:table-cell pl-2 py-3.5 text-left text-sm font-semibold text-gray-900">{i18n.str`Total credit`}</th>
              <th scope="col" class="hidden sm:table-cell pl-2 py-3.5 text-left text-sm font-semibold text-gray-900">{i18n.str`Status`}</th>
              <th scope="col" class="hidden sm:table-cell pl-2 py-3.5 text-left text-sm font-semibold text-gray-900">{i18n.str`Subject`}</th>
            </tr>
          </thead>
          <tbody>
            {Object.entries(txByDate).map(([date, txs], idx) => {
              return <Fragment key={idx}>
                <tr class="border-t border-gray-200">
                  <th colSpan={6} scope="colgroup" class="bg-gray-50 py-2 pl-4 pr-3 text-left text-sm font-semibold text-gray-900 sm:pl-3">
                    {date}
                  </th>
                </tr>
                {txs.map(item => {
                  const creationTime = item.creation_time.t_s === "never" ? "" : format(item.creation_time.t_s * 1000, "HH:mm:ss")
                  const confirmationTime = item.confirmation_time
                    ? item.confirmation_time.t_s === "never" ? i18n.str`never` : format(item.confirmation_time.t_s, "dd/MM/yyyy HH:mm:ss")
                    : "-"
                  return (<tr key={idx} class="border-b border-gray-200 hover:bg-gray-200 last:border-none">

                    <td onClick={(e) => {
                        e.preventDefault();
                        onSelected(item.id);
                      }} class="relative py-2 pl-2 pr-2 text-sm ">
                      <div class="font-medium text-gray-900">{creationTime}</div>
                      {/* <dl class="font-normal sm:hidden">
                        <dt class="sr-only sm:hidden"><i18n.Translate>Amount</i18n.Translate></dt>
                        <dd class="mt-1 truncate text-gray-700">
                          {item.negative ? i18n.str`sent` : i18n.str`received`} {item.amount ? (
                            <span data-negative={item.negative ? "true" : "false"} class="data-[negative=false]:text-green-600 data-[negative=true]:text-red-600">
                              <RenderAmount value={item.amount} />
                            </span>
                          ) : (
                            <span style={{ color: "grey" }}>&lt;{i18n.str`invalid value`}&gt;</span>
                          )}</dd>

                        <dt class="sr-only sm:hidden"><i18n.Translate>Counterpart</i18n.Translate></dt>
                        <dd class="mt-1 truncate text-gray-500 sm:hidden">
                          {item.negative ? i18n.str`to` : i18n.str`from`} {item.counterpart}
                        </dd>
                        <dd class="mt-1 text-gray-500 sm:hidden" >
                          <pre class="break-words w-56 whitespace-break-spaces p-2 rounded-md mx-auto my-2 bg-gray-100">
                            {item.subject}
                          </pre>
                        </dd>
                      </dl> */}
                    </td>
                    <td onClick={(e) => {
                        e.preventDefault();
                        onSelected(item.id);
                      }}class="hidden sm:table-cell px-3 py-3.5 text-sm text-gray-500 cursor-pointer">{confirmationTime}</td>
                    <td onClick={(e) => {
                        e.preventDefault();
                        onSelected(item.id);
                      }}class="hidden sm:table-cell px-3 py-3.5 text-sm text-red-600 cursor-pointer"><RenderAmount value={Amounts.parseOrThrow(item.amount_debit)} spec={resp.body.regional_currency_specification} /></td>
                    <td onClick={(e) => {
                        e.preventDefault();
                        onSelected(item.id);
                      }}class="hidden sm:table-cell px-3 py-3.5 text-sm text-green-600 cursor-pointer"><RenderAmount value={Amounts.parseOrThrow(item.amount_credit)} spec={resp.body.fiat_currency_specification} /></td>

                    <td  onClick={(e) => {
                        e.preventDefault();
                        onSelected(item.id);
                      }}class="hidden sm:table-cell px-3 py-3.5 text-sm text-gray-500 cursor-pointer">{item.status}</td>
                    <td onClick={(e) => {
                        e.preventDefault();
                        onSelected(item.id);
                      }} class="hidden sm:table-cell px-3 py-3.5 text-sm text-gray-500 break-all min-w-md">
                      {item.subject}
                    </td>
                  </tr>)
                })}
              </Fragment>

            })}
          </tbody>

        </table>


      </div>
    </div>
  );

}
