import { TranslatedString, UnwrapKeyError } from "@gnu-taler/taler-util";
import { notifyError, notifyInfo, useTranslationContext } from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { createNewForm } from "../handlers/forms.js";

export function UnlockAccount({
  onAccountUnlocked,
  onRemoveAccount,
}: {
  onAccountUnlocked: (password: string) => Promise<void>;
  onRemoveAccount: () => void;
}): VNode {
  const { i18n } = useTranslationContext()
  const Form = createNewForm<{
    password: string;
  }>();

  return (
    <div class="flex min-h-full flex-col ">
      <div class="sm:mx-auto sm:w-full sm:max-w-md">
        <h1 class="mt-6 text-center text-2xl font-bold leading-9 tracking-tight text-gray-900">
          <i18n.Translate>Account locked</i18n.Translate>
        </h1>
        <p class="mt-6 text-lg leading-8 text-gray-600">
          <i18n.Translate>Your account is normally locked anytime you reload. To unlock type
            your password again.</i18n.Translate>
        </p>
      </div>

      <div class="mt-10 sm:mx-auto sm:w-full sm:max-w-[480px] ">
        <div class="bg-gray-100 px-6 py-6 shadow sm:rounded-lg sm:px-12">
          <Form.Provider
            initialValue={{}}
            onSubmit={async (v) => {
              try {
                await onAccountUnlocked(v.password!);
                notifyInfo("Account unlocked" as TranslatedString);
              } catch (e) {
                if (e instanceof UnwrapKeyError) {
                  notifyError(
                    "Could not unlock account" as any,
                    e.message as any,
                  );
                } else {
                  throw e;
                }
              }
            }}
          >
            <div class="mb-4">
              <Form.InputLine
                label={"Password" as TranslatedString}
                name="password"
                type="password"
                required
              />
            </div>

            <div class="mt-8">
              <button
                type="submit"
                class="flex w-full justify-center rounded-md bg-indigo-600 px-3 py-1.5 text-sm font-semibold leading-6 text-white shadow-sm hover:bg-indigo-500 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600"
              >
                <i18n.Translate>Unlock</i18n.Translate>
              </button>
            </div>
          </Form.Provider>
        </div>
        <button
          type="button"
          onClick={() => {
            onRemoveAccount();
          }}
          class="m-4 block rounded-md bg-red-600 px-3 py-2 text-center text-sm  text-white shadow-sm hover:bg-red-500 "
        >
          <i18n.Translate>Forget account</i18n.Translate>
        </button>
      </div>
    </div>
  );
}
