import { TalerError, TalerExchangeApi, TranslatedString, assertUnreachable } from "@gnu-taler/taler-util";
import { ErrorLoading, Loading, useTranslationContext } from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { useState } from "preact/hooks";
import { createNewForm } from "../handlers/forms.js";
import { useCases } from "../hooks/useCases.js";
import { Pages } from "../pages.js";
import { AmlExchangeBackend } from "../types.js";

import { Officer } from "./Officer.js";
import { amlStateConverter } from "./ShowConsolidated.js";

export function CasesUI({ records, filter, onChangeFilter, onFirstPage, onNext }: { onFirstPage?: () => void, onNext?: () => void, filter: AmlExchangeBackend.AmlState, onChangeFilter: (f: AmlExchangeBackend.AmlState) => void, records: TalerExchangeApi.AmlRecord[] }): VNode {
  const { i18n } = useTranslationContext();

  const form = createNewForm<{ state: AmlExchangeBackend.AmlState }>();

  return <div>
    <div class="sm:flex sm:items-center">
      <div class="px-2 sm:flex-auto">
        <h1 class="text-base font-semibold leading-6 text-gray-900">
          <i18n.Translate>
            Cases
          </i18n.Translate>
        </h1>
        <p class="mt-2 text-sm text-gray-700">
          <i18n.Translate>
            A list of all the account with the status
          </i18n.Translate>
        </p>
      </div>
      <div class="px-2">
        <form.Provider
          initialValue={{ state: filter }}
          onUpdate={(v) => {
            onChangeFilter(v.state ?? filter);
          }}
          onSubmit={(v) => { }}
        >
          <form.InputChoiceHorizontal
            name="state"
            label={i18n.str`Filter`}
            converter={amlStateConverter}
            choices={[
              {
                label: "Pending" as TranslatedString,
                value: AmlExchangeBackend.AmlState.pending,
              },
              {
                label: "Frozen" as TranslatedString,
                value: AmlExchangeBackend.AmlState.frozen,
              },
              {
                label: "Normal" as TranslatedString,
                value: AmlExchangeBackend.AmlState.normal,
              },
            ]}
          />

        </form.Provider>
      </div>
    </div>
    <div class="mt-8 flow-root">
      <div class="overflow-x-auto">
        {!records.length ? (
          <div>empty result </div>
        ) : (
          <div class="inline-block min-w-full py-2 align-middle sm:px-6 lg:px-8">
            <table class="min-w-full divide-y divide-gray-300">
              <thead>
                <tr>
                  <th
                    scope="col"
                    class="px-3 py-3.5 text-left text-sm font-semibold text-gray-900"
                  >
                    <i18n.Translate>
                      Account Id
                    </i18n.Translate>
                  </th>
                  <th
                    scope="col"
                    class="px-3 py-3.5 text-left text-sm font-semibold text-gray-900"
                  >
                    <i18n.Translate>
                      Status
                    </i18n.Translate>
                  </th>
                  <th
                    scope="col"
                    class="sm:hidden px-3 py-3.5 text-left text-sm font-semibold text-gray-900"
                  >
                    <i18n.Translate>
                      Threshold
                    </i18n.Translate>
                  </th>
                </tr>
              </thead>
              <tbody class="divide-y divide-gray-200 bg-white">
                {records.map((r) => {
                  return (
                    <tr class="hover:bg-gray-100 ">
                      <td class="whitespace-nowrap px-3 py-5 text-sm text-gray-500 ">
                        <div class="text-gray-900">
                          <a
                            href={Pages.account.url({ account: r.h_payto })}
                            class="text-indigo-600 hover:text-indigo-900"
                          >
                            {r.h_payto.substring(0, 16)}...
                          </a>
                        </div>
                      </td>
                      <td class="whitespace-nowrap px-3 py-5 text-sm text-gray-500">
                        {((state: AmlExchangeBackend.AmlState): VNode => {
                          switch (state) {
                            case AmlExchangeBackend.AmlState.normal: {
                              return (
                                <span class="inline-flex items-center rounded-md bg-green-50 px-2 py-1 text-xs font-medium text-green-700 ring-1 ring-inset ring-green-600/20">
                                  Normal
                                </span>
                              );
                            }
                            case AmlExchangeBackend.AmlState.pending: {
                              return (
                                <span class="inline-flex items-center rounded-md bg-yellow-50 px-2 py-1 text-xs font-medium text-yellow-700 ring-1 ring-inset ring-green-600/20">
                                  Pending
                                </span>
                              );
                            }
                            case AmlExchangeBackend.AmlState.frozen: {
                              return (
                                <span class="inline-flex items-center rounded-md bg-red-50 px-2 py-1 text-xs font-medium text-red-700 ring-1 ring-inset ring-green-600/20">
                                  Frozen
                                </span>
                              );
                            }
                          }
                        })(r.current_state)}
                      </td>
                      <td class="whitespace-nowrap px-3 py-5 text-sm text-gray-900">
                        {r.threshold}
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
            <Pagination onFirstPage={onFirstPage} onNext={onNext} />
          </div>
        )}
      </div>
    </div>
  </div>

}


export function Cases() {
  const [stateFilter, setStateFilter] = useState(AmlExchangeBackend.AmlState.pending);

  const list = useCases(stateFilter);

  if (!list) {
    return <Loading />
  }
  if (list instanceof TalerError) {
    return <ErrorLoading error={list} />
  }

  if (list.data.type === "fail") {
    switch (list.data.case) {
      case "unauthorized": return <Officer />
      case "officer-not-found": return <Officer />
      case "officer-disabled": return <Officer />
      default: assertUnreachable(list.data)
    }
  }

  const { records } = list.data.body

  return <CasesUI
    records={records}
    onFirstPage={list.pagination && !list.pagination.isFirstPage ? list.pagination.reset : undefined}
    onNext={list.pagination && !list.pagination.isLastPage ? list.pagination.loadMore : undefined}
    filter={stateFilter}
    onChangeFilter={setStateFilter}
  />
}

export const PeopleIcon = () => <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="w-6 h-6">
  <path stroke-linecap="round" stroke-linejoin="round" d="M15.75 6a3.75 3.75 0 11-7.5 0 3.75 3.75 0 017.5 0zM4.501 20.118a7.5 7.5 0 0114.998 0A17.933 17.933 0 0112 21.75c-2.676 0-5.216-.584-7.499-1.632z" />
</svg>

export const HomeIcon = () => <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="w-6 h-6">
  <path stroke-linecap="round" stroke-linejoin="round" d="M2.25 12l8.954-8.955c.44-.439 1.152-.439 1.591 0L21.75 12M4.5 9.75v10.125c0 .621.504 1.125 1.125 1.125H9.75v-4.875c0-.621.504-1.125 1.125-1.125h2.25c.621 0 1.125.504 1.125 1.125V21h4.125c.621 0 1.125-.504 1.125-1.125V9.75M8.25 21h8.25" />
</svg>


export const ChevronRightIcon = () => <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="w-6 h-6">
  <path stroke-linecap="round" stroke-linejoin="round" d="M8.25 4.5l7.5 7.5-7.5 7.5" />
</svg>


export const ArrowRightIcon = () => <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="w-6 h-6">
  <path stroke-linecap="round" stroke-linejoin="round" d="M13.5 4.5L21 12m0 0l-7.5 7.5M21 12H3" />
</svg>


function Pagination({ onFirstPage, onNext }: { onFirstPage?: () => void, onNext?: () => void, }) {
  const { i18n } = useTranslationContext()
  return (
    <nav class="flex items-center justify-between border-t border-gray-200 bg-white px-4 py-3 sm:px-6 rounded-lg" aria-label="Pagination">
      <div class="flex flex-1 justify-between sm:justify-end">
        <button
          class="relative disabled:bg-gray-100 disabled:text-gray-500 inline-flex items-center rounded-md bg-white px-3 py-2 text-sm font-semibold text-gray-900 ring-1 ring-inset ring-gray-300 hover:bg-gray-50 focus-visible:outline-offset-0"
          disabled={!onFirstPage}
          onClick={onFirstPage}
        >
          <i18n.Translate>First page</i18n.Translate>
        </button>
        <button
          class="relative disabled:bg-gray-100 disabled:text-gray-500 ml-3 inline-flex items-center rounded-md bg-white px-3 py-2 text-sm font-semibold text-gray-900 ring-1 ring-inset ring-gray-300 hover:bg-gray-50 focus-visible:outline-offset-0"
          disabled={!onNext}
          onClick={onNext}
        >
          <i18n.Translate>Next</i18n.Translate>
        </button>
      </div>
    </nav>

  )
}
