import {
  AbsoluteTime,
  Codec,
  LockedAccount,
  OfficerAccount,
  OfficerId,
  SigningKey,
  buildCodecForObject,
  codecForAbsoluteTime,
  codecForString,
  codecOptional,
  createNewOfficerAccount,
  decodeCrock,
  encodeCrock,
  unlockOfficerAccount,
} from "@gnu-taler/taler-util";
import {
  buildStorageKey,
  useLocalStorage,
  useMemoryStorage,
} from "@gnu-taler/web-util/browser";
import { useMemo } from "preact/hooks";

export interface Officer {
  account: LockedAccount;
  when: AbsoluteTime;
}

const codecForLockedAccount = codecForString() as Codec<LockedAccount>;

type OfficerAccountString = {
  id: string,
  strKey: string;
}

export const codecForOfficerAccount = (): Codec<OfficerAccountString> =>
  buildCodecForObject<OfficerAccountString>()
  .property("id", codecForString()) // FIXME
  .property("strKey", codecForString()) // FIXME
  .build("OfficerAccount");

export const codecForOfficer = (): Codec<Officer> =>
  buildCodecForObject<Officer>()
    .property("account", codecForLockedAccount) // FIXME
    .property("when", codecForAbsoluteTime) // FIXME
    .build("Officer");

export type OfficerState = OfficerNotReady | OfficerReady;
export type OfficerNotReady = OfficerNotFound | OfficerLocked;
interface OfficerNotFound {
  state: "not-found";
  create: (password: string) => Promise<void>;
}
interface OfficerLocked {
  state: "locked";
  forget: () => void;
  tryUnlock: (password: string) => Promise<void>;
}
interface OfficerReady {
  state: "ready";
  account: OfficerAccount;
  forget: () => void;
  lock: () => void;
}

const OFFICER_KEY = buildStorageKey("officer", codecForOfficer());
const DEV_ACCOUNT_KEY = buildStorageKey("account-dev", codecForOfficerAccount());
const ACCOUNT_KEY = "account";

export function useOfficer(): OfficerState {
  // dev account, is save when reloaded.
  const accountStorage = useLocalStorage(DEV_ACCOUNT_KEY);
  const account = useMemo(() => {
    if (!accountStorage.value) return undefined
    return {
      id: accountStorage.value.id as OfficerId,
      signingKey: decodeCrock(accountStorage.value.strKey) as SigningKey
    }
  }, [accountStorage.value])

  
  // const accountStorage = useMemoryStorage<OfficerAccount>(ACCOUNT_KEY);
  // const account = accountStorage.value;

  const officerStorage = useLocalStorage(OFFICER_KEY);  
  const officer = officerStorage.value;
  

  if (officer === undefined) {
    return {
      state: "not-found",
      create: async (pwd: string) => {
        const { id, safe, signingKey } = await createNewOfficerAccount(pwd);        
        officerStorage.update({
          account: safe,
          when: AbsoluteTime.now(),
        });

        // accountStorage.update({ id, signingKey });
        const strKey = encodeCrock(signingKey)
        accountStorage.update({id, strKey })
      },
    };
  }

  if (account === undefined) {
    return {
      state: "locked",
      forget: () => {
        officerStorage.reset();
      },
      tryUnlock: async (pwd: string) => {
        const ac = await unlockOfficerAccount(officer.account, pwd);
        // accountStorage.update(ac);
        accountStorage.update({id: ac.id, strKey: encodeCrock(ac.signingKey)})
      },
    };
  }

  return {
    state: "ready",
    account,
    lock: () => {
      accountStorage.reset();
    },
    forget: () => {
      officerStorage.reset();
      accountStorage.reset();
    },
  };
}
