import { canonicalizeBaseUrl } from "@gnu-taler/taler-util";
import {
  HttpResponseOk,
  RequestOptions,
  useApiContext,
} from "@gnu-taler/web-util/browser";
import { useCallback } from "preact/hooks";
import { uiSettings } from "../settings.js";

interface useBackendType {
  request: <T>(
    path: string,
    options?: RequestOptions,
  ) => Promise<HttpResponseOk<T>>;
  fetcher: <T>(args: [string, string]) => Promise<HttpResponseOk<T>>;
  paginatedFetcher: <T>(
    args: [string, number, number, string],
  ) => Promise<HttpResponseOk<T>>;
}
export function usePublicBackend(): useBackendType {
  const { request: requestHandler } = useApiContext();

  const baseUrl = getInitialBackendBaseURL();

  const request = useCallback(
    function requestImpl<T>(
      path: string,
      options: RequestOptions = {},
    ): Promise<HttpResponseOk<T>> {
      return requestHandler<T>(baseUrl, path, options);
    },
    [baseUrl],
  );

  const fetcher = useCallback(
    function fetcherImpl<T>([endpoint, talerAmlOfficerSignature]: [string, string]): Promise<HttpResponseOk<T>> {
      return requestHandler<T>(baseUrl, endpoint, {
        talerAmlOfficerSignature
      });
    },
    [baseUrl],
  );
  const paginatedFetcher = useCallback(
    function fetcherImpl<T>([endpoint, page, size, talerAmlOfficerSignature]: [
      string,
      number,
      number,
      string,
    ]): Promise<HttpResponseOk<T>> {
      return requestHandler<T>(baseUrl, endpoint, {
        params: { page: page || 1, size },
        talerAmlOfficerSignature,
      });
    },
    [baseUrl],
  );
  return {
    request,
    fetcher,
    paginatedFetcher,
  };
}

export function getInitialBackendBaseURL(): string {
  const overrideUrl =
    typeof localStorage !== "undefined"
      ? localStorage.getItem("exchange-base-url")
      : undefined;

  let result: string;

  if (!overrideUrl) {
    //normal path
    if (!uiSettings.backendBaseURL) {
      console.error(
        "ERROR: backendBaseURL was overridden by a setting file and missing. Setting value to 'window.origin'",
      );      
      result = typeof (window as any) !== "undefined" ? window.origin : "localhost"
    } else {
      result = uiSettings.backendBaseURL;
    }
  } else {
    // testing/development path
    result = overrideUrl
  }
  try {
    return canonicalizeBaseUrl(result)
  } catch (e) {
    //fall back
    return canonicalizeBaseUrl(window.origin)
  }
}
