import { AbsoluteTime, TranslatedString } from "@gnu-taler/taler-util";
import { FormState } from "../handlers/FormProvider.js";
import { BaseForm } from "../pages/AntiMoneyLaunderingForm.js";
import { FlexibleForm, languageList } from "./index.js";
import { resolutionSection } from "./simplest.js";

export const v1 = (current: BaseForm): FlexibleForm<Form902_1.Form> => ({
  design: [
    {
      title: "Information on customer" as TranslatedString,
      description:
        "The customer is the person with whom the member concludes the contract with regard to the financial service provided (civil law). Does the member act as director of a domiciliary company, this domiciliary company is the customer." as TranslatedString,
      fields: [
        {
          type: "choiceStacked",
          props: {
            name: "customerType",
            label: "Type of customer" as TranslatedString,
            required: true,
            choices: [
              {
                label: "Natural person" as TranslatedString,
                value: "natural",
              },
              {
                label: "Legal entity" as TranslatedString,
                value: "legal",
              },
            ],
          },
        },
        {
          type: "text",
          props: {
            name: "naturalCustomer.fullName",
            label: "Full name" as TranslatedString,
            required: true,
          },
        },
        {
          type: "text",
          props: {
            name: "naturalCustomer.address",
            label: "Residential address" as TranslatedString,
            required: true,
          },
        },
        {
          type: "integer",
          props: {
            name: "naturalCustomer.telephone",
            label: "Telephone" as TranslatedString,
          },
        },
        {
          type: "text",
          props: {
            name: "naturalCustomer.email",
            label: "E-mail" as TranslatedString,
          },
        },
        {
          type: "date",
          props: {
            name: "naturalCustomer.dateOfBirth",
            label: "Date of birth" as TranslatedString,
            required: true,
            // help: "format 'dd/MM/yyyy'" as TranslatedString,
          },
        },
        {
          type: "text",
          props: {
            name: "naturalCustomer.nationality",
            label: "Nationality" as TranslatedString,
            required: true,
          },
        },
        {
          type: "text",
          props: {
            name: "naturalCustomer.document",
            label: "Identification document" as TranslatedString,
            required: true,
          },
        },
        {
          type: "file",
          props: {
            name: "naturalCustomer.documentAttachment",
            label: "Document attachment" as TranslatedString,
            required: true,
            maxBites: 2 * 1024 * 1024,
            accept: ".png",
            help: "Max size of 2 mega bytes" as TranslatedString,
          },
        },
        {
          type: "text",
          props: {
            name: "naturalCustomer.companyName",
            label: "Company name" as TranslatedString,
          },
        },
        {
          type: "text",
          props: {
            name: "naturalCustomer.office",
            label: "Registered office" as TranslatedString,
          },
        },
        {
          type: "text",
          props: {
            name: "naturalCustomer.companyDocument",
            label: "Company identification document" as TranslatedString,
          },
        },
        {
          type: "file",
          props: {
            name: "naturalCustomer.companyDocumentAttachment",
            label: "Document attachment" as TranslatedString,
            required: true,
            maxBites: 2 * 1024 * 1024,
            accept: ".png",
            help: "Max size of 2 mega bytes" as TranslatedString,
          },
        },
        {
          type: "text",
          props: {
            name: "legalCustomer.companyName",
            label: "Company name" as TranslatedString,
            required: true,
          },
        },
        {
          type: "text",
          props: {
            name: "legalCustomer.domicile",
            label: "Domicile" as TranslatedString,
            required: true,
          },
        },
        {
          type: "text",
          props: {
            name: "legalCustomer.contactPerson",
            label: "Contact person" as TranslatedString,
          },
        },
        {
          type: "text",
          props: {
            name: "legalCustomer.telephone",
            label: "Telephone" as TranslatedString,
          },
        },
        {
          type: "text",
          props: {
            name: "legalCustomer.email",
            label: "E-mail" as TranslatedString,
          },
        },
        {
          type: "text",
          props: {
            name: "legalCustomer.document",
            label: "Identification document" as TranslatedString,
            help: "Not older than 12 month" as TranslatedString,
          },
        },
        {
          type: "file",
          props: {
            name: "legalCustomer.documentAttachment",
            label: "Document attachment" as TranslatedString,
            required: true,
            maxBites: 2 * 1024 * 1024,
            accept: ".png",
            help: "Max size of 2 mega bytes" as TranslatedString,
          },
        },
      ],
    },
    {
      title:
        "Information on the natural persons who establish the business relationship for legal entities and partnerships" as TranslatedString,
      description:
        "For legal entities and partnerships the identity of the natural persons who establish the business relationship must be verified." as TranslatedString,
      fields: [
        {
          type: "array",
          props: {
            name: "businessEstablisher",
            label: "Persons" as TranslatedString,
            required: true,
            placeholder: "this is the placeholder" as TranslatedString,
            fields: [
              {
                type: "text",
                props: {
                  name: "fullName",
                  label: "Full name" as TranslatedString,
                  required: true,
                },
              },
              {
                type: "text",
                props: {
                  name: "address",
                  label: "Residential address" as TranslatedString,
                  required: true,
                },
              },
              {
                type: "date",
                props: {
                  name: "dateOfBirth",
                  label: "Date of birth" as TranslatedString,
                  required: true,
                  // help: "format 'dd/MM/yyyy'" as TranslatedString,
                },
              },
              {
                type: "text",
                props: {
                  name: "nationality",
                  label: "Nationality" as TranslatedString,
                  required: true,
                },
              },
              {
                type: "text",
                props: {
                  name: "typeOfAuthorization",
                  label:
                    "Type of authorization (signatory of representation)" as TranslatedString,
                  required: true,
                },
              },
              {
                type: "file",
                props: {
                  name: "documentAttachment",
                  label:
                    "Identification document attachment" as TranslatedString,
                  required: true,
                  maxBites: 2 * 1024 * 1024,
                  accept: ".png",
                  help: "Max size of 2 mega bytes" as TranslatedString,
                },
              },
              {
                type: "choiceStacked",
                props: {
                  name: "powerOfAttorneyArrangements",
                  label: "Power of attorney arrangements" as TranslatedString,
                  required: true,
                  choices: [
                    {
                      label: "CR extract" as TranslatedString,
                      value: "cr",
                    },
                    {
                      label: "Mandate" as TranslatedString,
                      value: "mandate",
                    },
                    {
                      label: "Other" as TranslatedString,
                      value: "other",
                    },
                  ],
                },
              },
              {
                type: "text",
                props: {
                  name: "powerOfAttorneyArrangementsOther",
                  label: "Power of attorney arrangements" as TranslatedString,
                  required: true,
                },
              },
            ],
            labelField: "fullName",
          },
        },
      ],
    },
    {
      title: "Acceptance of business relationship" as TranslatedString,
      fields: [
        {
          type: "date",
          props: {
            name: "acceptance.when",
            pattern: "dd/MM/yyyy",
            label: "Date (conclusion of contract)" as TranslatedString,
            // help: "format 'dd/MM/yyyy'" as TranslatedString,
          },
        },
        {
          type: "choiceStacked",
          props: {
            name: "acceptance.acceptedBy",
            label: "Accepted by" as TranslatedString,
            required: true,
            choices: [
              {
                label: "Face-to-face meeting with customer" as TranslatedString,
                value: "face-to-face",
              },
              {
                label:
                  "Correspondence: authenticated copy of identification document obtained" as TranslatedString,
                value: "correspondence-document",
              },
              {
                label:
                  "Correspondence: residential address validated" as TranslatedString,
                value: "correspondence-address",
              },
            ],
          },
        },
        {
          type: "choiceStacked",
          props: {
            name: "acceptance.typeOfCorrespondence",
            label: "Type of correspondence service" as TranslatedString,
            choices: [
              {
                label: "to the customer" as TranslatedString,
                value: "customer",
              },
              {
                label: "hold at bank" as TranslatedString,
                value: "bank",
              },
              {
                label: "to the member" as TranslatedString,
                value: "member",
              },
              {
                label: "to a third party" as TranslatedString,
                value: "third-party",
              },
            ],
          },
        },
        {
          type: "text",
          props: {
            name: "acceptance.thirdPartyFullName",
            label: "Third party full name" as TranslatedString,
            required: true,
          },
        },
        {
          type: "text",
          props: {
            name: "acceptance.thirdPartyAddress",
            label: "Third party  address" as TranslatedString,
            required: true,
          },
        },
        {
          type: "selectMultiple",
          props: {
            name: "acceptance.language",
            label: "Languages" as TranslatedString,
            choices: languageList,
            unique: true,
          },
        },
        {
          type: "textArea",
          props: {
            name: "acceptance.furtherInformation",
            label: "Further information" as TranslatedString,
          },
        },
      ],
    },
    {
      title:
        "Information on the beneficial owner of the assets and/or controlling person" as TranslatedString,
      description:
        "Establishment of the beneficial owner of the assets and/or controlling person" as TranslatedString,
      fields: [
        {
          type: "choiceStacked",
          props: {
            name: "establishment",
            label: "The customer is" as TranslatedString,
            required: true,
            choices: [
              {
                label:
                  "a natural person and there are no doubts that this person is the sole beneficial owner of the assets" as TranslatedString,
                value: "natural",
              },
              {
                label:
                  "a foundation (or a similar construct; incl. underlying companies)" as TranslatedString,
                value: "foundation",
              },
              {
                label:
                  "a trust (incl. underlying companies)" as TranslatedString,
                value: "trust",
              },
              {
                label:
                  "a life insurance policy with separately managed accounts/securities accounts" as TranslatedString,
                value: "insurance-wrapper",
              },
              {
                label: "all other cases" as TranslatedString,
                value: "other",
              },
            ],
          },
        },
      ],
    },
    {
      title:
        "Evaluation with regard to embargo procedures/terrorism lists on establishing the business relationship" as TranslatedString,
      description:
        "Verification whether the customer, beneficial owners of the assets, controlling persons, authorized representatives or other involved persons are listed on an embargo/terrorism list (date of verification/result)" as TranslatedString,
      fields: [
        {
          type: "textArea",
          props: {
            name: "embargoEvaluation",
            help: "The evaluation must be made at the beginning of the business relationship and has to be repeated in the case of permanent business relationship every time the according lists are updated." as TranslatedString,
            label: "Evaluation" as TranslatedString,
          },
        },
      ],
    },
    {
      title:
        "In the case of cash transactions/occasional customers: Information on type and purpose of business relationship" as TranslatedString,
      description:
        "These details are only necessary for occasional customers, i.e. money exchange, money and asset transfer or other cash transactions provided that no customer profile (VQF doc. No. 902.5) is created" as TranslatedString,
      fields: [
        {
          type: "choiceStacked",
          props: {
            name: "cashTransactions.typeOfBusiness",
            label: "Type of business relationship" as TranslatedString,
            choices: [
              {
                label: "Money exchange" as TranslatedString,
                value: "money-exchange",
              },
              {
                label: "Money and asset transfer" as TranslatedString,
                value: "money-and-asset-transfer",
              },
              {
                label:
                  "Other cash transactions. Specify below" as TranslatedString,
                value: "other",
              },
            ],
          },
        },
        {
          type: "text",
          props: {
            name: "cashTransactions.otherTypeOfBusiness",
            required: true,
            label: "Specify other cash transactions:" as TranslatedString,
          },
        },
        {
          type: "textArea",
          props: {
            name: "cashTransactions.purpose",
            label:
              "Purpose of the business relationship (purpose of service requested)" as TranslatedString,
          },
        },
      ],
    },
    resolutionSection(current),
  ],
  behavior: function formBehavior(
    v: Partial<Form902_1.Form>,
  ): FormState<Form902_1.Form> {
    return {
      fullName: {
        disabled: true,
      },
      businessEstablisher: {
        elements: (v.businessEstablisher ?? []).map((be) => {
          return {
            powerOfAttorneyArrangementsOther: {
              hidden: be.powerOfAttorneyArrangements !== "other",
            },
          };
        }),
      },
      acceptance: {
        thirdPartyFullName: {
          hidden: v.acceptance?.typeOfCorrespondence !== "third-party",
        },
        thirdPartyAddress: {
          hidden: v.acceptance?.typeOfCorrespondence !== "third-party",
        },
      },
      cashTransactions: {
        otherTypeOfBusiness: {
          hidden: v.cashTransactions?.typeOfBusiness !== "other",
        },
      },
      naturalCustomer: {
        fullName: {
          hidden: v.customerType !== "natural",
        },
        address: {
          hidden: v.customerType !== "natural",
        },
        telephone: {
          hidden: v.customerType !== "natural",
        },
        email: {
          hidden: v.customerType !== "natural",
        },
        dateOfBirth: {
          hidden: v.customerType !== "natural",
        },
        nationality: {
          hidden: v.customerType !== "natural",
        },
        document: {
          hidden: v.customerType !== "natural",
        },
        companyName: {
          hidden: v.customerType !== "natural",
        },
        office: {
          hidden: v.customerType !== "natural",
        },
        companyDocument: {
          hidden: v.customerType !== "natural",
        },
        companyDocumentAttachment: {
          hidden: v.customerType !== "natural",
        },
        documentAttachment: {
          hidden: v.customerType !== "natural",
        },
      },
      legalCustomer: {
        companyName: {
          hidden: v.customerType !== "legal",
        },
        contactPerson: {
          hidden: v.customerType !== "legal",
        },
        document: {
          hidden: v.customerType !== "legal",
        },
        domicile: {
          hidden: v.customerType !== "legal",
        },
        email: {
          hidden: v.customerType !== "legal",
        },
        telephone: {
          hidden: v.customerType !== "legal",
        },
        documentAttachment: {
          hidden: v.customerType !== "legal",
        },
      },
    };
  },
});

namespace Form902_1 {
  interface LegalEntityCustomer {
    companyName: string;
    domicile: string;
    contactPerson: string;
    telephone: string;
    email: string;
    document: string;
    documentAttachment: string;
  }
  interface NaturalCustomer {
    fullName: string;
    address: string;
    telephone: string;
    email: string;
    dateOfBirth: AbsoluteTime;
    nationality: string;
    document: string;
    documentAttachment: string;
    companyName: string;
    office: string;
    companyDocument: string;
    companyDocumentAttachment: string;
  }

  interface Person {
    fullName: string;
    address: string;
    dateOfBirth: AbsoluteTime;
    nationality: string;
    typeOfAuthorization: string;
    document: string;
    documentAttachment: string;
    powerOfAttorneyArrangements: "cr" | "mandate" | "other";
    powerOfAttorneyArrangementsOther: string;
  }

  interface Acceptance {
    when: AbsoluteTime;
    acceptedBy: "face-to-face" | "authenticated-copy";
    typeOfCorrespondence: string;
    language: string[];
    furtherInformation: string;
    thirdPartyFullName: string;
    thirdPartyAddress: string;
  }

  interface BeneficialOwner {
    establishment:
    | "natural-person"
    | "foundation"
    | "trust"
    | "insurance-wrapper"
    | "other";
  }

  interface CashTransactions {
    typeOfBusiness: "money-exchange" | "money-and-asset-transfer" | "other";
    otherTypeOfBusiness: string;
    purpose: string;
  }

  export interface Form extends BaseForm {
    fullName: string;
    customerType: "natural" | "legal";
    naturalCustomer: NaturalCustomer;
    legalCustomer: LegalEntityCustomer;
    businessEstablisher: Array<Person>;
    acceptance: Acceptance;
    beneficialOwner: BeneficialOwner;
    embargoEvaluation: string;
    cashTransactions: CashTransactions;
    // enclosures: Enclosures;
  }
}
